/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.wizard;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import javax.ide.Service;
import javax.ide.command.Context;
import javax.ide.extension.ExtensionRegistry;
import javax.ide.spi.ProviderNotFoundException;
import javax.ide.util.MetaClass;
import javax.ide.wizard.spi.WizardHook;

/**
 * The WizardManager provides the interface through which the IDE invokes 
 * Wizards.<p>
 */
public class WizardManager extends Service
{
  private final Map _wizardInstancesById = new HashMap();

  
  /**
   * Invokes a wizard with the specified execution context and
   * parameters.<p>
   *
   * @param context The execution context for the invokable.
   * @param id The id identifying the wizard.
   *
   * @return <code>true</code> if the wizard completed successfully,
   * otherwise <code>false</code>.
   *
   */
  public boolean invokeWizard( Context context, String id )
  {
    Wizard theWizard = findOrCreateWizard( id );
    return theWizard.invoke( context );
  }
  
  /**
   * Initialize the wizard manager.
   */
  protected final WizardHook getWizardHook()
  {
    return (WizardHook)ExtensionRegistry.getExtensionRegistry().getHook( WizardHook.ELEMENT );
  }

  private MetaClass getWizardClass( String id )
  {
    return getWizardHook().getWizardInfo( id ).getWizardClass();
  }
  
  /**
   * Get the class names of all registered wizards.
   * 
   * @return a collection of string class names of all registered wizards. 
   */
  protected final Collection /*<String>*/ getAllWizardClasses()
  {
    return getWizardHook().getAllWizardClasses();
  }
  
  /**
   * Find or create a wizard instance.
   * 
   * @param id the id of the wizard to find or create.
   * @return an existing instance of the Wizard or a new instance if none has
   *    been created.
   *    
   * @throws IllegalArgumentException if no such wizard is registered.
   * @throws IllegalStateException if the wizard could not be created.
   */
  protected final Wizard findOrCreateWizard( String id )
  {
    Wizard theWizard = (Wizard) _wizardInstancesById.get( id );
    if ( theWizard == null )
    {
      MetaClass wizardClass = getWizardClass( id );
      if ( wizardClass == null )
      {
        throw new IllegalArgumentException( "No such wizard: " + id );
      }
      try
      {
        theWizard = (Wizard) wizardClass.newInstance();
        _wizardInstancesById.put( id, theWizard );
      }
      catch ( Exception e )
      {
        throw new IllegalStateException( "Failed to create wizard " + id );
      }
    }
    return theWizard;
  }

  /**
   * Get the wizard manager implementation for this IDE.
   * 
   * @return the wizard manager implementation for this ide.
   */
  public static WizardManager getWizardManager()
  {
    try
    {
      return (WizardManager) getService( WizardManager.class ); 
    }
    catch ( ProviderNotFoundException lnfe )
    {
      lnfe.printStackTrace();
      throw new IllegalStateException( "no wizard manager" );
    }
  }
}
