/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.view;

import java.beans.PropertyChangeListener;
import javax.ide.util.IconDescription;

/**
 * <code>Viewable</code>s are objects that map directly to visible
 * GUI elements that users can select.
 */
public interface Viewable 
{
  /**
   * Identifies the bound property 'label'.
   */
  public static final String PROP_LABEL = "label";

  /**
   * Identifies the bound property 'iconPath'.
   */
  public static final String PROP_ICON_PATH = "iconPath";

  /**
   * Identifies the bound property 'visible'.
   */
  public static final String PROP_VISIBLE = "visible";

  /**
   * Get the object visible state.
   *
   * @return The command visible state.
   */
  public boolean isVisible();
  
  /**
   *  Returns a short label that can be displayed to the user.
   *  Generally, the value of the returned {@link String} is considered
   *  translatable and should therefore be placed in an appropriate
   *  resource file.  When possible, the returned label should be
   *  reasonably short enough to show in the navigator or explorer
   *  windows but long enough to clearly identify and distinguish the
   *  <CODE>Viewable</CODE>.
   *
   *  @return a short descriptive label of the <CODE>Viewable</CODE>
   *  that can be shown to the user.
   */
  public String getLabel();

  /**
   *  Returns the tool tip text to show when the mouse pointer pauses
   *  over a UI component that represents this <CODE>Viewable</CODE>.
   *
   *  @return the tooltip to show when the mouse pointer pauses over a
   *  UI component that represents this <CODE>Viewable</CODE>.
   */
  public String getToolTip();

  /**
   * Gets a relative path to the icon. Generally, the is considered 
   * translatable, therefore, the path value be placed in an appropriate 
   * resource file. The path must be relative the Viewable implementation 
   * class file location or the resource file location if one is provided.
   *
   * @return The icon relative path.
   */
  public IconDescription getIcon();

  /**
   * Returns the label.  This overrides the <code>toString</code> method in 
   * <code>java.lang.Object</code>. <p>
   * Implementors of the <code>Viewable</code> interface should 
   * override this as appropriate.  The default implementation is
   * the same as <code>getLabel()</code>
   *
   * @return A short label describing this <code>Viewable</code>.
   *
   * @see java.lang.Object#toString()
   * @see #getLabel()
   */
  public String toString();

  /**
   * Add a {@link PropertyChangeListener} to the listener list.
   * A <code>PropertyChangeEvent</code> will be fired in response to setting 
   * a bound property.
   * @param listener A {@link PropertyChangeListener}.
   */
  public void addPropertyChangeListener( PropertyChangeListener listener );

  /**
   * Removes a {@link PropertyChangeListener} from the listener list.
   * @param listener A {@link PropertyChangeListener}.
   */
  public void removePropertyChangeListener( PropertyChangeListener listener );
}
