/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.view;

import java.net.URI;

import javax.ide.command.Context;
import javax.ide.view.GUIPanel;

/**
 * The IDEDialogs provides the interface through which extension writers 
 * can invoke standard IDE dialogs, such as: information, warning, error, 
 * file and directory selection dialogs.
 */
public interface IDEDialogs 
{	
	/**
	 * Button id for an "Ok" button.
	 */
	public int OK_ID = 0;

	/**
	 * Button id for a "Yes" button.
	 */
	public int YES_ID = 1;

	/**
	 * Button id for a "No" button.
	 */
	public int NO_ID = 2;

	/**
	 * Button id for a "Cancel" button.
	 */
	public int CANCEL_ID = 3;

  /**
   * Requests an option dialog with a OK and Cancel buttons.
   */
  public int OK_CANCEL_OPTION = 0;

  /**
   * Requests an option dialog with a Yes and No buttons.
   */
  public int YES_NO_OPTION = 1;

  /**
   * Requests an option dialog with a Yes, No, and Cancel buttons.
   */
  public int YES_NO_CANCEL_OPTION = 2;

  /**
   * Creates a "file selection" dialog.
   * @param title The dialog title.
   * @param location The default starting directory or pre-selected file.
   * @param context The current {@link Context}.
   * @param parent The parent window handle {@link GUIPanel}
   * @return The newly created {@link URISelectionDialog} class
   */
  URISelectionDialog getFileSelectionDialog( String title,
                                             URI location, 
                                             Context context, 
                                             GUIPanel parent );
                                             

  /**
   * Creates a "directory selection" dialog.
   * @param title The dialog title.
   * @param location The default starting directory or pre-selected file.
   * @param context The current {@link Context}.
   * @param parent The parent window handle {@link GUIPanel}
   * @return The newly created {@link URISelectionDialog} class
   */
  URISelectionDialog getDirectorySelectionDialog( String title,
                                                  URI location, 
                                                  Context context, 
                                                  GUIPanel parent );

  //-----------------------------------------------------------------
  // Convience methods to show information dialogs
  //-----------------------------------------------------------------

  /**
   * Show an error message box with title and message as specified 
   * @param title The title of the error dialog
   * @param message The error message.
   * @param parent The parent window handle {@link GUIPanel}
   */    
  void showErrorDialog( String title, String message, GUIPanel parent );

  /**
   * Show an informational message box with title and message as specified.
   * @param title The message box title
   * @param message The message text.
   * @param parent The parent window handle {@link GUIPanel}
   */
  void showInformationDialog( String title, String message, GUIPanel parent );

  /**
   * Show a warning type dialog with title and message as specified.
   * @param title The title of the warning dialog
   * @param message The message text
   * @param parent The parent window handle {@link GUIPanel}
   */
  void showWarningDialog( String title, String message, GUIPanel parent );

  /**
   * Show an options dialog with title and message as specified. An option 
   * dialog can have OK, Cancel buttons; Yes, No buttons; and Yes, No, Cancel
   * buttons. The button option is controlled by the constants: 
   * OK_CANCEL_OPTION, YES_NO_OPTION, and YES_NO_CANCEL_OPTION.
   * @param title The question title
   * @param message The question text
   * @param option Button options. Can be one of the following constants:
   * OK_CANCEL_OPTION, YES_NO_OPTION, and YES_NO_CANCEL_OPTION.
   * @param parent The parent window handle {@link GUIPanel}
   * @return One of the following constants: OK_ID or YES_ID, NO_ID, CANCEL_ID.
   */
  int showOptionDialog( String title, 
                        String message, 
                        int option, 
                        GUIPanel parent );

}

