package javax.ide.util;

import java.io.IOException;
import java.io.InputStream;

import java.net.URL;

/**
 * Provides delayed loading for resources.
 * 
 * This is a utility class for storing resource and classloader pairs.
 * It can be used to store paths to resources without looking them up as in the 
 * case of images and properties file. It can also be used in cases where a 
 * given resource, e.g. an xml file, contains class references that should all 
 * resolve with the given classloader.
 * 
 * @author svassile
 * @see MetaClass 
 * 
 */
public final class MetaResource {
    private final String m_resourcePath;
    private final ClassLoader m_classLoader;
    private URL m_resourceURL;
    
    /**
     * Creates an MetaResource instance. Does not try to access the resource.
     * 
     * @param classLoader the classloader that should load the resource
     * @param path the relative path to the resource e.g. "images/foo.png"
     * @throws NullPointerException when either the classloader or the path are 
     * <code>null</code>.
     */
    public MetaResource(ClassLoader classLoader, String path) {
        if (classLoader == null)
        {
            throw new NullPointerException( "null classLoader" );    
        }
        if (path == null)
        {
            throw new NullPointerException( "null resource path" );        
        }
        m_classLoader = classLoader;
        m_resourcePath = path;
    }
    
    /**
     * Looks up a resource, starting at the classloader passed in. This method 
     * does not initialize the extension.
     * 
     * @return the url of the resource if it exists or <code>null</code> if it 
     * does not
     */
    public URL toURL() {
        if (m_resourceURL == null)
        {
            m_resourceURL = m_classLoader.getResource(m_resourcePath);
        }
        return m_resourceURL;
    }
    
    /**
     * Looks up a resource, and opens a resource stream on it starting at the 
     * classloader passed in. This method does not initialize the extension.
     * 
     * @return opens and returns an {@code InputStream} of the resource if it 
     * exists or <code>null</code> if it does not
     * 
     * @throws IOException if the openStream failed for whatever reason
     */
    public InputStream getInputStream() throws IOException {
        final URL url = toURL();
        if (url != null)
        {
            return url.openStream();
        }
        return null;
        
    }

    /**
     * Gets the resource path.
     * 
     * @return the resource path
     */
    public String getResourcePath() {
        return m_resourcePath;
    }

    /**
     * Gets the classloader.
     * 
     * @return the classloader
     */
    public ClassLoader getClassLoader() {
        return m_classLoader;
    }
}
