/*
 * @(#)TypeArgumentT.java
 */

package javax.ide.model.java.source.tree;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * A type argument for a type reference. </p>
 *
 * @author Andy Yu
 * */
public interface TypeArgumentT
  extends Tree
{
  // ----------------------------------------------------------------------

  public static final TypeArgumentT[] EMPTY_ARRAY =
    new TypeArgumentT[ 0 ];

  // ----------------------------------------------------------------------

  /**
   * Gets the bound kind. <p/>
   *
   * <pre>
   * EXACT. The exact type is specified.
   * EXTENDS. Type specifies an "extends" bound. Only types that
   *                extend the given bound are allowed.
   * SUPER. Type specifies a "super" bound. Only super types of
   *              the given bound are allowed.
   * UNBOUNDED. No explicit bound. Implies "extends" java/lang/Object.
   * </pre>
   */
  public BoundKind getBoundKind();

  /**
   * Gets the bound's type reference. <p/>
   *
   * <pre>
   * EXACT. Returns the specified type.
   * EXTENDS. Returns the "extends" bound.
   * SUPER. Returns the "super" bound.
   * UNBOUNDED. Returns null.
   * </pre>
   *
   * @return This element's type reference.
   */
  public TypeReferenceT getType();

  /**
   * Unlinks the current source type and links the input element.
   * 
   * @param type The new type reference.
   *
   * @throws IllegalStateException if the input type is already linked.
   */
  public void setType( TypeReferenceT type );


  // ----------------------------------------------------------------------

  /**
   * In this version, this class is 1.4 compatible. In a later version,
   * it will be redone as an enum. <p/>
   */
  public static final class BoundKind
  {
    private static final Map values = new LinkedHashMap();

    private static final int VALUE_EXACT = 0;
    private static final int VALUE_EXTENDS = 1;
    private static final int VALUE_SUPER = 2;
    private static final int VALUE_UNBOUNDED = 3;

    /** <code>T</code> */
    public static final BoundKind EXACT =
      new BoundKind( VALUE_EXACT, "EXACT" );

    /** <code>? extends T</code> */
    public static final BoundKind EXTENDS =
      new BoundKind( VALUE_EXTENDS, "EXTENDS" );

    /** <code>? super T</code> */
    public static final BoundKind SUPER =
      new BoundKind( VALUE_SUPER, "SUPER" );

    /** <code>?</code> */
    public static final BoundKind UNBOUNDED =
      new BoundKind( VALUE_UNBOUNDED, "UNBOUNDED" );


    // ----------------------------------------------------------------------

    private final int ordinal;
    
    private final String name;

    private BoundKind(int ordinal, String name)
    {
      this.ordinal = ordinal;
      this.name = name;
      
      values.put( name, this );
    }

    // ----------------------------------------------------------------------

    // Begin enum compatibility section.
    
    public String name()
    {
      return name;
    }
    
    public String toString()
    {
      return name();
    }
    
    public int ordinal()
    {
      return ordinal;
    }
    
    public int hashCode()
    {
      return ordinal();
    }
    
    public int compareTo(BoundKind other)
    {
      return ordinal() - other.ordinal();
    }
    
    public boolean equals(Object other)
    {
      if (other instanceof BoundKind)
      {
        final BoundKind tk = (BoundKind) other;
        return ordinal() == tk.ordinal();
      }
      
      return false;
    }

    public Class getDeclaringClass()
    {
      return BoundKind.class;
    }


    // ----------------------------------------------------------------------

    public static BoundKind valueOf(int ordinal)
    {
      return values()[ ordinal ];
    }

    public static BoundKind valueOf(Class ignored, String name)
    {
      return (BoundKind) values.get(name);
    }

    public static BoundKind[] values()
    {
      final Collection entries = values.values();
      return (BoundKind[])entries.toArray(new BoundKind[entries.size()]);
    }


    // ----------------------------------------------------------------------
  }
}
