/*
 * @(#)ExpressionT.java
 */

package javax.ide.model.java.source.tree;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Common supertypes for all expressions. If it is a primary,
 * selector, or method invocation, it has a name. Otherwise, it is an
 * operator of some sort and has operands. With each enumerated
 * constant is a description of which fields are non-null and
 * useful. <p/>
 *
 * @author Andy Yu
 * */
public interface ExpressionT
  extends Tree
{
  // ----------------------------------------------------------------------

  public static final ExpressionT[] EMPTY_ARRAY =
    new ExpressionT[ 0 ];


  // ----------------------------------------------------------------------

  /**
   * Identifies the operation this expression is performing.
   */
  public ExpressionKind getExpressionKind();


  // ----------------------------------------------------------------------

  /**
   * @return the ExpressionT representing the first operand.
   * Non-null if this expression is an operator (and hence has
   * operands).
   */
  public ExpressionT getFirstOperand();

  /**
   * @return the ExpressionT representing the second operand.
   * Non-null if this operator has a second operand.
   */
  public ExpressionT getSecondOperand();

  /**
   * @return the ExpressionT representing the third operand.
   * Non-null if this has a third operand, e.g. "x? x: x".
   */
  public ExpressionT getThirdOperand();

  /**
   * @return the array of ExpressionTs. Always non-null. May be
   * zero-length. Returns a collection of ExpressionTs. <p/>
   *
   * List of ExpressionTs.
   */
  public List getOperands();

  /**
   * @return The size of the operand array.
   */
  public int getOperandCount();

  /**
   * Gets the operand at the specified index in the operand array.
   *
   * @param index 0-based.
   *
   * @return Null if the indicated operand doesn't exist.
   */
  public ExpressionT getOperandAt( int index );


  // ----------------------------------------------------------------------

  /**
   * An enumeration identifying which kind of expression this is.
   */
  public static final class ExpressionKind
  {
    private static final Map values = new LinkedHashMap();

    private static final int VALUE_base = 0;
    private static final int VALUE_ADD = VALUE_base + 0;
    private static final int VALUE_ADDASG = VALUE_base + 1;
    private static final int VALUE_AND = VALUE_base + 2;
    private static final int VALUE_ANNOTATION = VALUE_base + 3;
    private static final int VALUE_ARRAYCONST = VALUE_base + 4;
    private static final int VALUE_ASG = VALUE_base + 5;
    private static final int VALUE_BITAND = VALUE_base + 6;
    private static final int VALUE_BITANDASG = VALUE_base + 7;
    private static final int VALUE_BITNOT = VALUE_base + 8;
    private static final int VALUE_BITOR = VALUE_base + 9;
    private static final int VALUE_BITORASG = VALUE_base + 10;
    private static final int VALUE_BITXOR = VALUE_base + 11;
    private static final int VALUE_BITXORASG = VALUE_base + 12;
    private static final int VALUE_CLASS_LITERAL = VALUE_base + 13;
    private static final int VALUE_COND = VALUE_base + 14;
    private static final int VALUE_DEREF = VALUE_base + 15;
    private static final int VALUE_DIV = VALUE_base + 16;
    private static final int VALUE_DIVASG = VALUE_base + 17;
    private static final int VALUE_DOT = VALUE_base + 18;
    private static final int VALUE_EQ = VALUE_base + 19;
    private static final int VALUE_GE = VALUE_base + 20;
    private static final int VALUE_GT = VALUE_base + 21;
    private static final int VALUE_IDENTIFIER = VALUE_base + 22;
    private static final int VALUE_INSTANCEOF = VALUE_base + 23;
    private static final int VALUE_INVOKE = VALUE_base + 24;
    private static final int VALUE_LE = VALUE_base + 25;
    private static final int VALUE_LIST = VALUE_base + 26;
    private static final int VALUE_LITERAL = VALUE_base + 27;
    private static final int VALUE_LSH = VALUE_base + 28;
    private static final int VALUE_LSHASG = VALUE_base + 29;
    private static final int VALUE_LT = VALUE_base + 30;
    private static final int VALUE_MINUS = VALUE_base + 31;
    private static final int VALUE_MINUSASG = VALUE_base + 32;
    private static final int VALUE_MOD = VALUE_base + 33;
    private static final int VALUE_MODASG = VALUE_base + 34;
    private static final int VALUE_MUL = VALUE_base + 35;
    private static final int VALUE_MULASG = VALUE_base + 36;
    private static final int VALUE_NEGATIVE = VALUE_base + 37;
    private static final int VALUE_NEWARRAY = VALUE_base + 38;
    private static final int VALUE_NEWOBJECT = VALUE_base + 39;
    private static final int VALUE_NOT = VALUE_base + 40;
    private static final int VALUE_NOTEQ = VALUE_base + 41;
    private static final int VALUE_OR = VALUE_base + 42;
    private static final int VALUE_POSITIVE = VALUE_base + 43;
    private static final int VALUE_POSTDEC = VALUE_base + 44;
    private static final int VALUE_POSTINC = VALUE_base + 45;
    private static final int VALUE_PREDEC = VALUE_base + 46;
    private static final int VALUE_PREINC = VALUE_base + 47;
    private static final int VALUE_QUALIFIED_SUPER = VALUE_base + 48;
    private static final int VALUE_QUALIFIED_THIS = VALUE_base + 49;
    private static final int VALUE_RSH = VALUE_base + 50;
    private static final int VALUE_RSHASG = VALUE_base + 51;
    private static final int VALUE_TYPE = VALUE_base + 52;
    private static final int VALUE_TYPECAST = VALUE_base + 53;
    private static final int VALUE_URSH = VALUE_base + 54;
    private static final int VALUE_URSHASG = VALUE_base + 55;
    private static final int VALUE_WRAPPER = VALUE_base + 56;
    private static final int VALUE_max = VALUE_base + 57;


    // ----------------------------------------------------------------------

    /** Arithmetic addition or string concatenation. 2+ operands. */
    public static final ExpressionKind EXPR_ADD =
      new ExpressionKind( VALUE_ADD, "EXPR_ADD", InfixExpressionT.class );

    /** Arithmetic addition or string concatenation assignment. 2
     * operands. */
    public static final ExpressionKind EXPR_ADDASG =
      new ExpressionKind( VALUE_ADDASG, "EXPR_ADDASG",
                          AssignmentExpressionT.class );

    /** Conditional (logical) AND. 2+ operands. */
    public static final ExpressionKind EXPR_AND =
      new ExpressionKind( VALUE_AND, "EXPR_AND", InfixExpressionT.class );

    /** Annotation expression. 0 operands. */
    public static final ExpressionKind EXPR_ANNOTATION =
      new ExpressionKind( VALUE_ANNOTATION, "EXPR_ANNOTATION",
                          AnnotationExpressionT.class );

    /** An array constant. 0+ operands. This is LIST rather than
     * LITERAL because this is not a lexer literal. */
    public static final ExpressionKind EXPR_ARRAYCONST =
      new ExpressionKind( VALUE_ARRAYCONST, "EXPR_ARRAYCONST",
                          ListExpressionT.class );

    /** Assignment. 2 operands. */
    public static final ExpressionKind EXPR_ASG =
      new ExpressionKind( VALUE_ASG, "EXPR_ASG", AssignmentExpressionT.class );

    /** Bitwise AND. 2+ operands. */
    public static final ExpressionKind EXPR_BITAND =
      new ExpressionKind( VALUE_BITAND, "EXPR_BITAND",
                          InfixExpressionT.class );

    /** Bitwise AND assignment. 2 operands. */
    public static final ExpressionKind EXPR_BITANDASG =
      new ExpressionKind( VALUE_BITANDASG, "EXPR_BITANDASG",
                          AssignmentExpressionT.class );

    /** Bitwise NOT (negation or complement). 2 operands. */
    public static final ExpressionKind EXPR_BITNOT =
      new ExpressionKind( VALUE_BITNOT, "EXPR_BITNOT",
                          UnaryExpressionT.class );

    /** Bitwise OR. 2+ operands. */
    public static final ExpressionKind EXPR_BITOR =
      new ExpressionKind( VALUE_BITOR, "EXPR_BITOR", InfixExpressionT.class );

    /** Bitwise OR assignment. 2 operands. */
    public static final ExpressionKind EXPR_BITORASG =
      new ExpressionKind( VALUE_BITORASG, "EXPR_BITORASG",
                          AssignmentExpressionT.class );
    
    /** Bitwise XOR. 2+ operands. */
    public static final ExpressionKind EXPR_BITXOR =
      new ExpressionKind( VALUE_BITXOR, "EXPR_BITXOR",
                          InfixExpressionT.class );
    
    /** Bitwise XOR assignment. 2 operands. */
    public static final ExpressionKind EXPR_BITXORASG =
      new ExpressionKind( VALUE_BITXORASG, "EXPR_BITXORASG",
                          AssignmentExpressionT.class );

    /** A class literal. 1 operand. Operand is an TYPE. Despite the
     * name, this is not a lexical literal. We call it a class literal
     * because that's what the JLS calls it. */
    public static final ExpressionKind EXPR_CLASS_LITERAL =
      new ExpressionKind( VALUE_CLASS_LITERAL, "EXPR_CLASS_LITERAL",
                          UnaryExpressionT.class );

    /** Conditional operator (?:). 3 operands. */
    public static final ExpressionKind EXPR_COND =
      new ExpressionKind( VALUE_COND, "EXPR_COND", QuestionExpressionT.class );

    /** Array dereference. 2 operands. 1st is the expression lhs. 2nd
     * is LIST with an operand per array dereference. */
    public static final ExpressionKind EXPR_DEREF =
      new ExpressionKind( VALUE_DEREF, "EXPR_DEREF",
                          ArrayAccessExpressionT.class );

    /** Arithmetic division. 2 operands. */
    public static final ExpressionKind EXPR_DIV =
      new ExpressionKind( VALUE_DIV, "EXPR_DIV", InfixExpressionT.class );

    /** Arithmetic division assignment. 2 operands. */
    public static final ExpressionKind EXPR_DIVASG =
      new ExpressionKind( VALUE_DIVASG, "EXPR_DIVASG",
                          AssignmentExpressionT.class );

    /** Dot dereference. 1 operand. */
    public static final ExpressionKind EXPR_DOT =
      new ExpressionKind( VALUE_DOT, "EXPR_DOT", DotExpressionT.class );

    /** Equal. 2 operands. */
    public static final ExpressionKind EXPR_EQ =
      new ExpressionKind( VALUE_EQ, "EXPR_EQ", InfixExpressionT.class );

    /** Greater than or equal to. 2 operands. */
    public static final ExpressionKind EXPR_GE =
      new ExpressionKind( VALUE_GE, "EXPR_GE", InfixExpressionT.class );

    /** Greater than. 2 operands. */
    public static final ExpressionKind EXPR_GT =
      new ExpressionKind( VALUE_GT, "EXPR_GT", InfixExpressionT.class );

    /** Simple name. 0 operands. */
    public static final ExpressionKind EXPR_IDENTIFIER =
      new ExpressionKind( VALUE_IDENTIFIER, "EXPR_IDENTIFIER",
                          IdentifierExpressionT.class );

    /** Instanceof. 2 operands. 2nd is an TYPE. */
    public static final ExpressionKind EXPR_INSTANCEOF =
      new ExpressionKind( VALUE_INSTANCEOF, "EXPR_INSTANCEOF",
                          InfixExpressionT.class );

    /** Method invocation. 1, 2, or 3 operands. If 1, the operand is
     * the LIST arguments. If 2, the 1st operand is the lhs and the
     * 2nd is the LIST arguments. If 3, the 1st operand is the lhs,
     * the 2nd is the type arguments, and the 3rd is the LIST
     * arguments. In all cases, the last operand is the LIST
     * arguments. */
    public static final ExpressionKind EXPR_INVOKE =
      new ExpressionKind( VALUE_INVOKE, "EXPR_INVOKE",
                          MethodCallExpressionT.class );

    /** Less than or equal to. 2 operands. */
    public static final ExpressionKind EXPR_LE =
      new ExpressionKind( VALUE_LE, "EXPR_LE", InfixExpressionT.class );

    /** A list of operands. 0+ operands. */
    public static final ExpressionKind EXPR_LIST =
      new ExpressionKind( VALUE_LIST, "EXPR_LIST", ListExpressionT.class );

    /** Lexical literal. 0 operands. */
    public static final ExpressionKind EXPR_LITERAL =
      new ExpressionKind( VALUE_LITERAL, "EXPR_LITERAL",
                          LiteralExpressionT.class );

    /** Bitwise left shift. 2 operands. */
    public static final ExpressionKind EXPR_LSH =
      new ExpressionKind( VALUE_LSH, "EXPR_LSH", InfixExpressionT.class );

    /** Bitwise left shift assignment. 2 operands. */
    public static final ExpressionKind EXPR_LSHASG =
      new ExpressionKind( VALUE_LSHASG, "EXPR_LSHASG", AssignmentExpressionT.class );

    /** Less than. 2 operands. */
    public static final ExpressionKind EXPR_LT =
      new ExpressionKind( VALUE_LT, "EXPR_LT", InfixExpressionT.class );

    /** Arithmetic subtraction. 2 operands. */
    public static final ExpressionKind EXPR_MINUS =
      new ExpressionKind( VALUE_MINUS, "EXPR_MINUS", InfixExpressionT.class );

    /** Arithmetic subtraction assignment. 2 operands. */
    public static final ExpressionKind EXPR_MINUSASG =
      new ExpressionKind( VALUE_MINUSASG, "EXPR_MINUSASG", 
                          AssignmentExpressionT.class );

    /** Arithmetic remainder (modulus). 2 operands. */
    public static final ExpressionKind EXPR_MOD =
      new ExpressionKind( VALUE_MOD, "EXPR_MOD", InfixExpressionT.class );

    /** Arithmetic remainder (modulus) assignment. 2 operands. */
    public static final ExpressionKind EXPR_MODASG =
      new ExpressionKind( VALUE_MODASG, "EXPR_MODASG",
                          AssignmentExpressionT.class );

    /** Arithmetic multiplication. 2+ operands. */
    public static final ExpressionKind EXPR_MUL =
      new ExpressionKind( VALUE_MUL, "EXPR_MUL", InfixExpressionT.class );

    /** Arithmetic multiplication assignment. 2 operands. */
    public static final ExpressionKind EXPR_MULASG =
      new ExpressionKind( VALUE_MULASG, "EXPR_MULASG",
                          AssignmentExpressionT.class );

    /** Arithmetic negative prefix. 1 operand. */
    public static final ExpressionKind EXPR_NEGATIVE =
      new ExpressionKind( VALUE_NEGATIVE, "EXPR_NEGATIVE",
                          UnaryExpressionT.class );

    /** Array creator. 1 operand. If operand is an LIST, then this is
     * an uninitialized array creator. Else the operand is an
     * ARRAYCONST and this is an initialized array creator. The type
     * involves a special getter. */
    public static final ExpressionKind EXPR_NEWARRAY =
      new ExpressionKind( VALUE_NEWARRAY, "EXPR_NEWARRAY",
                          NewArrayExpressionT.class );

    /** Class creator. 1 or 2 operands.  If 2, the 1st operand is the
     * lhs of an inner class creator. The last operand is an LIST for
     * the arguments. The type involves a special getter. The class
     * body involves a special getter. */
    public static final ExpressionKind EXPR_NEWOBJECT =
      new ExpressionKind( VALUE_NEWOBJECT, "EXPR_NEWOBJECT",
                          NewClassExpressionT.class );

    /** Logical NOT (negation or complement). 1 operand. */
    public static final ExpressionKind EXPR_NOT =
      new ExpressionKind( VALUE_NOT, "EXPR_NOT", UnaryExpressionT.class );

    /** Not-equal. 2 operands. */
    public static final ExpressionKind EXPR_NOTEQ =
      new ExpressionKind( VALUE_NOTEQ, "EXPR_NOTEQ", InfixExpressionT.class );

    /** Conditional (logical) OR. 2+ operands. */
    public static final ExpressionKind EXPR_OR =
      new ExpressionKind( VALUE_OR, "EXPR_OR", InfixExpressionT.class );

    /** Arithmetic positive prefix. 1 operand. */
    public static final ExpressionKind EXPR_POSITIVE =
      new ExpressionKind( VALUE_POSITIVE, "EXPR_POSITIVE",
                          UnaryExpressionT.class );

    /** Decrement postfix. 1 operand. */
    public static final ExpressionKind EXPR_POSTDEC =
      new ExpressionKind( VALUE_POSTDEC, "EXPR_POSTDEC",
                          UnaryExpressionT.class );

    /** Increment postfix. 1 operand. */
    public static final ExpressionKind EXPR_POSTINC =
      new ExpressionKind( VALUE_POSTINC, "EXPR_POSTINC",
                          UnaryExpressionT.class );

    /** Decrement prefix. 1 operand. */
    public static final ExpressionKind EXPR_PREDEC =
      new ExpressionKind( VALUE_PREDEC, "EXPR_PREDEC",
                          UnaryExpressionT.class );

    /** Increment prefix. 1 operand. */
    public static final ExpressionKind EXPR_PREINC =
      new ExpressionKind( VALUE_PREINC, "EXPR_PREINC",
                          UnaryExpressionT.class );

    /** Qualified super. 1 operand. Operand is an TYPE. */
    public static final ExpressionKind EXPR_QUALIFIED_SUPER =
      new ExpressionKind( VALUE_QUALIFIED_SUPER, "EXPR_QUALIFIED_SUPER",
                          UnaryExpressionT.class );

    /** Qualified this. 1 operand. Operand is an TYPE. */
    public static final ExpressionKind EXPR_QUALIFIED_THIS =
      new ExpressionKind( VALUE_QUALIFIED_THIS, "EXPR_QUALIFIED_THIS",
                          UnaryExpressionT.class );

    /** Bitwise signed right shift. 2 operands. */
    public static final ExpressionKind EXPR_RSH =
      new ExpressionKind( VALUE_RSH, "EXPR_RSH", InfixExpressionT.class );

    /** Bitwise signed right shift assignment. 2 operands. */
    public static final ExpressionKind EXPR_RSHASG =
      new ExpressionKind( VALUE_RSHASG, "EXPR_RSHASG",
                          AssignmentExpressionT.class );

    /** Type expression. 0 operands. */
    public static final ExpressionKind EXPR_TYPE =
      new ExpressionKind( VALUE_TYPE, "EXPR_TYPE", TypeExpressionT.class );

    /** Typecast. 2 operands. 1st is an TYPE. */
    public static final ExpressionKind EXPR_TYPECAST =
      new ExpressionKind( VALUE_TYPECAST, "EXPR_TYPECAST",
                          TypecastExpressionT.class );

    /** Bitwise unsigned right shift. 2 operands. */
    public static final ExpressionKind EXPR_URSH =
      new ExpressionKind( VALUE_URSH, "EXPR_URSH", InfixExpressionT.class );

    /** Bitwise unsigned right shift assignment. 2 operands. */
    public static final ExpressionKind EXPR_URSHASG =
      new ExpressionKind( VALUE_URSHASG, "EXPR_URSHASG",
                          AssignmentExpressionT.class );
    
    /** Nested expression (in parentheses or brackets). 1 operand. */
    public static final ExpressionKind EXPR_WRAPPER =
      new ExpressionKind( VALUE_WRAPPER, "EXPR_WRAPPER",
                          WrapperExpressionT.class );


    // ----------------------------------------------------------------------

    private final int ordinal;
    
    private final String name;

    private final Class exprClass;

    private ExpressionKind(int ordinal, String name, Class c)
    {
      this.ordinal = ordinal;
      this.name = name;
      this.exprClass = c;

      values.put( name, this );
    }

    public Class getExpressionClass()
    {
      return exprClass;
    }


    // ----------------------------------------------------------------------

    // Begin enum compatibility section.
    
    public String name()
    {
      return name;
    }
    
    public String toString()
    {
      return name();
    }
    
    public int ordinal()
    {
      return ordinal;
    }
    
    public int hashCode()
    {
      return ordinal();
    }
    
    public int compareTo(ExpressionKind other)
    {
      return ordinal() - other.ordinal();
    }
    
    public boolean equals(Object other)
    {
      if (other instanceof ExpressionKind)
      {
        final ExpressionKind tk = (ExpressionKind) other;
        return ordinal() == tk.ordinal();
      }
      
      return false;
    }

    public Class getDeclaringClass()
    {
      return ExpressionKind.class;
    }


    // ----------------------------------------------------------------------

    public static ExpressionKind valueOf(int ordinal)
    {
      return values()[ ordinal ];
    }

    public static ExpressionKind valueOf(Class ignored, String name)
    {
      return (ExpressionKind) values.get(name);
    }

    public static ExpressionKind[] values()
    {
      final Collection entries = values.values();
      return (ExpressionKind[])
        entries.toArray(new ExpressionKind[entries.size()]);
    }


    // ----------------------------------------------------------------------
  }
}
