/*
 * @(#)ClassT.java
 */

package javax.ide.model.java.source.tree;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * A type declaration. These include: "class" type, "interface" type,
 * "enum" type, "annotation" type. <p/>
 *
 * @author Andy Yu
 * */
public interface ClassT
  extends MemberT, HasNameT, BlockElementT
{
  // ----------------------------------------------------------------------

  public static final ClassT[] EMPTY_ARRAY = new ClassT[ 0 ];


  // ----------------------------------------------------------------------

  /**
   * Identifies what kind of class declaration this is.
   *
   * @return The JavaConstants.TY_* constant for this class.
   */
  public ClassKind getTypeKind();

  /**
   * Attempts to change the type of class this is.
   * <pre>
   * TY_CLASS: "class".
   * TY_ENUM: "enum".
   * TY_INTERFACE: "interface".
   * TY_ANNOTATE: "@interface".
   */
  public void setTypeKind( ClassKind typeKind );

  /**
   * True if this is a class or an enum type.  Backward compatible.
   * Checks against the presence of ACC_INTERFACE.
   *
   * @return True if this is a class or enum type. False if this is an
   * interface or annotation type.
   */
  public boolean isClass();

  /**
   * True if this is an interface or an annotation type.  Backward
   * compatible. Checks for the presence of ACC_INTERFACE.
   *
   * @return True if this is an interface or annotation type. False if
   * this is a class or enum type.
   */
  public boolean isInterface();

  /**
   * True if this is an enum type.  Checks for the presence of
   * ACC_ENUM. If true, then isClass() should also be true.
   *
   * @return True if this is an enum type. False otherwise.
   */
  public boolean isEnum();

  /**
   * True if this is an annotation type.  Checks for the presence of
   * ACC_ANNOTATION. If true, then isInterface() should also be true.
   *
   * @return True if this is an annotation type. False otherwise.
   */
  public boolean isAnnotation();


  // ----------------------------------------------------------------------

  /**
   * Gets the owning package declaration, null if none..
   * This amounts to calling getOwningSourceFile().getPackage().
   * 
   * @return The package declaration element.
   */
  public PackageT getPackageDeclaration();

  /**
   * Gets the owning package name, empty if none.
   *
   * Provided for convenience. This amounts to calling
   * getOwningSourceFile().getPackageName() with a null check.
   *
   * @return Gets the owning package name, empty if none. Always non-null.
   */
  public String getPackageName();


  // ----------------------------------------------------------------------

  /**
   * True if this is an inner class. Member, local, and anonymous classes
   * are all considered to be inner classes.
   * 
   * @return True if this is an inner class. False otherwise.
   */
  public boolean isInnerClass();

  /**
   * @return True if this is a top-level class or a member class.
   * False otherwise.
   */
  public boolean isExported();

  /**
   * True if this is an anonymous class declaration. Anonymous classes
   * return true for isClass().
   *
   * @return True if this is an anonymous class.
   */
  public boolean isAnonymous();


  // ----------------------------------------------------------------------

  /**
   * Gets the superclass declaration. On a "class" type, this will be
   * the "extends" clause.
   *
   * @return The super clause belonging to this class. Always non-null.
   * If there is no superclass clause, then a synthetic one is returned.
   */
  public SuperclassT getSuperclassClause();

  /**
   * Gets the interfaces declaration. On a "class" or "enum" type, this will
   * be the "implements" clause. On an "interface" type, this will be the
   * "extends" clause.
   *
   * @return The super clause belonging to this class. Always non-null.
   * If there is no interfaces clause, then a synthetic one is returned.
   */
  public InterfacesT getInterfacesClause();

  /**
   * Gets the class body.
   *
   * @return The class body belonging to this class. Always non-null.
   * If there is no class body, then a synthetic one is returned. Note:
   * It is a compile error if there is no class body.
   */
  public ClassBodyT getClassBody();


  // ----------------------------------------------------------------------

  /**
   * Gets the ordered list of type parameters declared on this class.
   * Remember, type parameters are not inherited by subclasses (and
   * subinterfaces). <p/>
   * 
   * For classes, syntax is "javadoc mods class name <ty_params> {}".
   *
   * @return The list of type parameters. <p/>
   *
   * List of TypeParameterTs.
   */
  public List getTypeParameters();

  /**
   * Gets the declared superclass. On a "class" type, this will be the
   * "extends" type.
   *
   * @return The explicitly declared superclass. Will be non-null if a
   * superclass is declared, even if the superclass cannot be legally
   * resolved. Will be null if no superclass is declared regardless of
   * the implicit superclass declaration.
   */
  public TypeReferenceT getSuperclass();

  /**
   * Attempts to set the declared base class of this class.
   *
   * @throws UnsupportedOperationException if this class may not have a
   *                                       superclass.
   * @throws IllegalStateException if the input element is already linked.
   */
  public void setSuperclass( TypeReferenceT superclass );

  /**
   * Gets the list of declared interfaces. Do not confuse this with
   * inner classes that are "interface" types. The name was chosen to be
   * consistent with JavaType and java/lang/Class. <p/>
   * 
   * On a "class" or "enum" type, this will be the "implements" types. On an
   * "interface" type, this will be the "extends" types.
   * 
   * @return The list of explicitly declared interfaces. Will have an element
   * for each super interface that is declared, even if the super interface
   * cannot be legally resolved. <p/>
   *
   * List of TypeReferenceTs.
   */
  public List getInterfaces();

  /**
   * Gets the list of declared member declarations.
   *
   * @return All declared members of this class, not including synthetic
   * members (e.g. default constructor). This includes: methods, constuctors,
   * field declarations, enum constant declarations, member classes, and
   * class initializers. <p/>
   *
   * List of MemberTs.
   */
  public List getDeclaredMembers();

  /**
   * Gets the collection of declared field variables and enum
   * constants.
   *
   * @return All declared field variables and enum constants, not
   * including synthetic ones (e.g. "this$0"). <p/>
   *
   * Collection of MemberVariableTs.
   */
  public Collection getDeclaredMemberVariables();

  /**
   * Gets the matching declared field variable or enum constant, null if none.
   * 
   * @param name The field or enum constant name to match. Require non-null.
   *
   * @return The first matching source-model declared field or enum
   * constant. Null if none.
   */
  public MemberVariableT getDeclaredMemberVariable( String name );

  /**
   * Gets the list of enum constants.
   *
   * Collection of EnumConstantTs.
   */
  public List getDeclaredEnumConstants();

  /**
   * Gets the list of declared field declarations. Does NOT include
   * enum constant declarations.
   *
   * @return All declared field declarations, not including synthetic
   * ones (e.g. "this$0"). This does NOT includes enum constant
   * declarations. <p/>
   *
   * List of FieldDeclTs.
   */
  public List getDeclaredFieldDeclarations();

  /**
   * Gets the collection of declared field variables. Does NOT include
   * enum constant variables.
   *
   * @return All declared field variables, not including synthetic
   * ones (e.g. "this$0"). This does NOT includes enum constant
   * variables. <p/>
   *
   * Collection of FieldVariableTs.
   */
  public Collection getDeclaredFieldVariables();

  /**
   * Gets the list of declared method (but not constructor)
   * declarations.
   *
   * @return All declared methods, not including constructors and not
   * including synthetic ones (e.g. "values()", "<clinit>()"). <p/>
   *
   * List of MethodTs.
   */
  public List getDeclaredMethods();

  /**
   * Gets the collection of matching methods (but not constructor)
   * declarations. This list will not be filtered according to method
   * signature. In other words, if there are multiple methods listed
   * with the same signature, this list will include them all.
   *
   * @param name The method name to match. Require non-null.
   *
   * @return All matching declared methods, but not constructors and not
   * synthetic ones (e.g. "values()", "<clinit>()"). <p/>
   *
   * Collection of MethodTs.
   */
  public Collection getDeclaredMethods( String name );

  /**
   * Gets the matching method declaration, null if none. This list will
   * not be filtered according to method signature. In other words, if
   * multiple methods are listed with the same signature, this list will
   * include them all.
   *
   * @param name Require non-null.
   * @param targetTypes Null indicates an empty parameter list.
   *
   * @return The first matching declared source-model method (but not
   * constructors). Null if none.
   */
//  public MethodT getDeclaredMethod( String name, JavaType[] targetTypes );

  /**
   * Gets the list of declared constructor declarations.
   *
   * @return All declared constructor declarations, not including 
   * synthetic ones. <p/>
   *
   * List of MethodTs.
   */
  public List getDeclaredConstructors();

  /**
   * Gets the matching declared constructor declaration, null if none.
   *
   * @param targetTypes Null indicates an empty parameter list.
   *
   * @return The first matching declared source-model constructor.
   * Null if none.
   */
//  public MethodT getDeclaredConstructor( JavaType[] targetTypes );

  /**
   * Gets the list of member class declarations. To be clear, this
   * list does not include local or anonymous classes.
   *
   * @return All declared member classes. <p/>
   *
   * List of ClassTs.
   */
  public List getDeclaredClasses();

  /**
   * Gets the matching member class declarations. To be clear, this will
   * not return a local or anonymous class.
   *
   * @param name The class name to match.
   *
   * @return The first matching member class.
   */
  public ClassT getDeclaredClass( String name );

  /**
   * Gets the list of declared class initializers. To be clear,
   * this list does NOT include the implicit block elements generated
   * for member variable initializers.
   *
   * @return All initializers declared in this class, not including
   * synthetic ones. <p/>
   *
   * List of ClassInitializerTs.
   */
  public List getDeclaredInitializers();


  // ----------------------------------------------------------------------

  /**
   * Gets the owning member declaration, null if none. <p/>
   *
   * The Java VM spec 3rd ed requires that local and anonymous classes
   * have an attribute specifying the enclosing method. At a source level,
   * we'll return the owning member. <p/>
   *
   * @return The directly enclosing member. If the directly enclosing
   * member is a class, then this will return the same thing as
   * getOwningClass(). If this is a top-level class, this will return null.
   */
  public MemberT getOwningMember();


  // ----------------------------------------------------------------------

  /**
   * An enumeration identifying which kind of type declaration this is.
   */
  public static final class ClassKind
  {
    private static final Map values = new LinkedHashMap();

    private static final int CLASS_VALUE = 0;
    private static final int INTERFACE_VALUE = 1;
    private static final int ENUM_VALUE = 2;
    private static final int ANNOTATION_VALUE = 3;

    /** This ClassT is of the "class" type, not an enum type. */
    public static final ClassKind CLASS_TYPE =
      new ClassKind( CLASS_VALUE, "CLASS_TYPE" );
    
    /** This ClassT is of the "interface" type, not an annotation type. */
    public static final ClassKind INTERFACE_TYPE =
      new ClassKind( INTERFACE_VALUE, "INTERFACE_TYPE" );
    
    /** This ClassT is of the "enum" type. */
    public static final ClassKind ENUM_TYPE =
      new ClassKind( ENUM_VALUE, "ENUM_TYPE" );
    
    /** This ClassT is of the "annotation" type. */
    public static final ClassKind ANNOTATION_TYPE =
      new ClassKind( ANNOTATION_VALUE, "ANNOTATION_TYPE" );
  

    // ----------------------------------------------------------------------

    private final int ordinal;
    
    private final String name;

    private ClassKind(int ordinal, String name)
    {
      this.ordinal = ordinal;
      this.name = name;
      
      values.put( name, this );
    }

    // ----------------------------------------------------------------------

    // Begin enum compatibility section.
    
    public String name()
    {
      return name;
    }
    
    public String toString()
    {
      return name();
    }
    
    public int ordinal()
    {
      return ordinal;
    }
    
    public int hashCode()
    {
      return ordinal();
    }
    
    public int compareTo(ClassKind other)
    {
      return ordinal() - other.ordinal();
    }
    
    public boolean equals(Object other)
    {
      if (other instanceof ClassKind)
      {
        final ClassKind tk = (ClassKind) other;
        return ordinal() == tk.ordinal();
      }
      
      return false;
    }

    public Class getDeclaringClass()
    {
      return ClassKind.class;
    }


    // ----------------------------------------------------------------------

    public static ClassKind valueOf(int ordinal)
    {
      return values()[ ordinal ];
    }

    public static ClassKind valueOf(Class ignored, String name)
    {
      return (ClassKind) values.get(name);
    }

    public static ClassKind[] values()
    {
      final Collection entries = values.values();
      return (ClassKind[])entries.toArray(new ClassKind[entries.size()]);
    }


    // ----------------------------------------------------------------------

  }
}
