/*
 * @(#)TreeFactory.java
 */

package javax.ide.model.java.source;

import java.lang.reflect.Modifier;
import javax.ide.model.java.source.tree.AnnotationT;
import javax.ide.model.java.source.tree.ArrayAccessExpressionT;
import javax.ide.model.java.source.tree.AssertStatementT;
import javax.ide.model.java.source.tree.AssignmentExpressionT;
import javax.ide.model.java.source.tree.BlockElementT;
import javax.ide.model.java.source.tree.BlockStatementT;
import javax.ide.model.java.source.tree.BlockT;
import javax.ide.model.java.source.tree.BreakStatementT;
import javax.ide.model.java.source.tree.CatchClauseT;
import javax.ide.model.java.source.tree.ClassBodyT;
import javax.ide.model.java.source.tree.ClassInitializerT;
import javax.ide.model.java.source.tree.ClassT;
import javax.ide.model.java.source.tree.ContinueStatementT;
import javax.ide.model.java.source.tree.DoStatementT;
import javax.ide.model.java.source.tree.DocCommentT;
import javax.ide.model.java.source.tree.DotExpressionT;
import javax.ide.model.java.source.tree.ElseClauseT;
import javax.ide.model.java.source.tree.EmptyStatementT;
import javax.ide.model.java.source.tree.ExpressionStatementT;
import javax.ide.model.java.source.tree.ExpressionT;
import javax.ide.model.java.source.tree.ExpressionT.ExpressionKind;
import javax.ide.model.java.source.tree.FieldDeclT;
import javax.ide.model.java.source.tree.FieldVariableT;
import javax.ide.model.java.source.tree.FinallyClauseT;
import javax.ide.model.java.source.tree.ForStatementT;
import javax.ide.model.java.source.tree.FormalParameterListT;
import javax.ide.model.java.source.tree.FormalParameterT;
import javax.ide.model.java.source.tree.IdentifierExpressionT;
import javax.ide.model.java.source.tree.IfStatementT;
import javax.ide.model.java.source.tree.ImportT;
import javax.ide.model.java.source.tree.InfixExpressionT;
import javax.ide.model.java.source.tree.InterfacesT;
import javax.ide.model.java.source.tree.ListExpressionT;
import javax.ide.model.java.source.tree.LocalVariableDeclT;
import javax.ide.model.java.source.tree.LocalVariableT;
import javax.ide.model.java.source.tree.MemberT;
import javax.ide.model.java.source.tree.MethodCallExpressionT;
import javax.ide.model.java.source.tree.MethodT;
import javax.ide.model.java.source.tree.NameT;
import javax.ide.model.java.source.tree.NewArrayExpressionT;
import javax.ide.model.java.source.tree.NewClassExpressionT;
import javax.ide.model.java.source.tree.PackageT;
import javax.ide.model.java.source.tree.QuestionExpressionT;
import javax.ide.model.java.source.tree.ReturnStatementT;
import javax.ide.model.java.source.tree.StatementT;
import javax.ide.model.java.source.tree.SuperclassT;
import javax.ide.model.java.source.tree.SwitchStatementT;
import javax.ide.model.java.source.tree.SynchStatementT;
import javax.ide.model.java.source.tree.ThrowStatementT;
import javax.ide.model.java.source.tree.ThrowsT;
import javax.ide.model.java.source.tree.TryStatementT;
import javax.ide.model.java.source.tree.TypeArgumentT;
import javax.ide.model.java.source.tree.TypeExpressionT;
import javax.ide.model.java.source.tree.TypeParameterT;
import javax.ide.model.java.source.tree.TypeReferenceT;
import javax.ide.model.java.source.tree.TypecastExpressionT;
import javax.ide.model.java.source.tree.UnaryExpressionT;
import javax.ide.model.java.source.tree.WhileStatementT;
import javax.ide.model.java.source.tree.WrapperExpressionT;

/**
 * Creates all Tree objects except for FileT objects. A TreeFactory is
 * owned by a specific FileT and all the Tree objects created by the
 * factory are owned by the same FileT. <p/>
 *
 * @author Andy Yu
 */
public abstract class TreeFactory
{
  // ----------------------------------------------------------------------
  //  G E N E R A L

  // ----------------------------------------------------------------------

  /**
   * Creates a name.
   *
   * @param name Require non-empty.
   */
  public abstract NameT createName( String name );

  /**
   * Creates a type reference.
   *
   * @param name Require non-null.
   */
  public final TypeReferenceT createTypeReference( String name )
  {
    return textToTypeReference( name );
  }

  /**
   * Creates a type reference.
   *
   * @param name Require non-null. Must be a simple name.
   * @param dimensions Require non-negative.
   */
  public abstract TypeReferenceT
    createTypeReference( String name, int dimensions );

  /**
   * Creates a type argument.
   *
   * @param bound Assuming the type is named "T", here is what you get
   *               for the indicated value of bound. <p/>
   *
   * <pre>
   *   BOUND_NARROW   T
   *   BOUND_EXTENDS  ? extends T
   *   BOUND_SUPER    ? super T
   *   BOUND_NONE     ?
   * </pre>
   *
   * @param type Ignored if bound == BOUND_NONE.
   */
  public abstract TypeArgumentT
    createTypeArgument( TypeArgumentT.BoundKind bound, TypeReferenceT type );


  // ----------------------------------------------------------------------
  //  T E X T - O N L Y   I N P U T S

  // ----------------------------------------------------------------------

  /**
   * Attempts to create a type with the given text.
   */
  public abstract AnnotationT textToAnnotation( String text );

  /**
   * Attempts to create a block with the given text.
   */
  public abstract BlockT textToBlock( String text );

  /**
   * Attempts to create a class with the given text.
   */
  public abstract ClassT textToClass( String text );

  /**
   * Attempts to create a doc comment with the given text.
   */
  public abstract DocCommentT textToDocComment( String text );

  /**
   * Attempts to create an expression with the given text.
   */
  public abstract ExpressionT textToExpression( String text );

  /**
   * Attempts to create a member with the given text. The input may be
   * a class, field, or method declaration. The input may also be a
   * class initializer.
   */
  public abstract MemberT textToMember( String text );

  /**
   * Attempts to create a block element with the given text.
   */
	public abstract BlockElementT textToBlockElement( String text );

  /**
   * Attempts to create a statement with the given text.
   */
  public abstract StatementT textToStatement( String text );

  /**
   * Attempts to create a type with the given text.
   */
  public abstract TypeReferenceT textToTypeReference( String text );


  // ----------------------------------------------------------------------
  //  D E C L A R A T I O N S

  // ----------------------------------------------------------------------

  /**
   * Creates an annotation (but not an annotation type). The
   * annotation type reference may not contain type arguments.
   *
   * @param name Require non-null.
   */
  public final AnnotationT createAnnotation( String name )
  {
    return createAnnotation( name, null );
  }

  /**
   * Creates an annotation (but not an annotation type).
   *
   * @param name Require non-null.
   * @param arguments Null indicates no argument list.
   */
  public abstract AnnotationT
    createAnnotation( String name, ListExpressionT arguments );

  /**
   * Creates a type parameter.
   *
   * @param name Require non-null.
   */
  public final TypeParameterT createTypeParameter( String name )
  {
    return createTypeParameter( name, TypeReferenceT.EMPTY_ARRAY );
  }

  /**
   * Creates a type parameter.
   *
   * @param name Require non-null.
   */
  public final TypeParameterT
    createTypeParameter( String name, TypeReferenceT bound )
  {
    return createTypeParameter( name, new TypeReferenceT[] { bound } );
  }

  /**
   * Creates a type parameter.
   *
   * @param name Require non-null.
   */
  public abstract TypeParameterT
    createTypeParameter( String name, TypeReferenceT[] bounds );


  // ----------------------------------------------------------------------

  /**
   * Creates a package declaration.
   *
   * @param name Require non-null.
   */
  public abstract PackageT createPackageDeclaration( String name );

  /**
   * Creates an import declaration.
   *
   * @param name Require non-null.
   */
  public abstract ImportT createImportDeclaration( String name );

  /**
   * Creates an import declaration.
   *
   * @param name Require non-null.
   */
  public abstract ImportT
    createStaticImportDeclaration( String name );

  /**
   * Creates a class declaration. Useful for those unfamiliar with jdk
   * 1.5.
   *
   * @param name Require non-null.
   * @param superclass Null indicates none.
   * @param interfaces Null indicates none.
   * @param body Null indicates an empty body.
   */
  public final ClassT
    createClass( ClassT.ClassKind typeKind,
								 String name,
                 SuperclassT superclass,
								 InterfacesT interfaces,
                 ClassBodyT body )
  {
		return createClass( typeKind, name, TypeParameterT.EMPTY_ARRAY,
												superclass, interfaces, body );
  }

  /**
   * Creates a class declaration.
   *
   * @param name Require non-null.
   * @param typeParameters Require non-null. For none, use
   *                     TypeParameterT.EMPTY_ARRAY.
   * @param superclass Null indicates none.
   * @param interfaces Null indicates none.
   * @param body Null indicates an empty body.
   */
  public abstract ClassT
    createClass( ClassT.ClassKind typeKind,
								 String name, 
                 TypeParameterT[] typeParameters,
								 SuperclassT superclass,
                 InterfacesT interfaces,
								 ClassBodyT body );


  // ----------------------------------------------------------------------

  /**
   * Creates a constructor declaration
   *
   * @param parameters Null indicates an empty parameter list.
   * @param throwsClause Null indicates no throws clause.
   * @param block Null indicates no an empty block.
   */
  public final MethodT
    createConstructor( FormalParameterListT parameters,
                       ThrowsT throwsClause,
                       BlockT block )
  {
    return createConstructor( null, parameters,
                              throwsClause, block );
  }

  /**
   * Creates a constructor declaration
   *
   * @param parameters Null indicates an empty parameter list.
   * @param throwsClause Null indicates no throws clause.
   * @param block Null indicates no an empty block.
   */
  public abstract MethodT
    createConstructor( TypeParameterT[] typeParameters,
                       FormalParameterListT parameters,
                       ThrowsT throwsClause,
                       BlockT block );

  /**
   * Creates a method declaration. Useful for those unfamiliar with
   * jdk 1.5.
   *
   * @param returnType Null indicates a void return type.
   * @param name Require non-null.
   * @param parameters Null indicates an empty parameter list.
   * @param throwsClause Null indicates no throws clause.
   * @param block Null indicates no block.
   */
  public final MethodT
    createMethod( TypeReferenceT returnType,
									String name,
									FormalParameterListT parameters,
									ThrowsT throwsClause,
									BlockT block )
  {
    return createMethod( TypeParameterT.EMPTY_ARRAY, returnType,
                         name, parameters, throwsClause, block );
  }

  /**
   * Creates a method declaration.
   *
   * @param typeParameters Require non-null. For none, use
   *                     TypeParameterT.EMPTY_ARRAY.
   * @param returnType Null indicates a void return type.
   * @param name Require non-null.
   * @param parameters Null indicates an empty parameter list.
   * @param throwsClause Null indicates no throws clause.
   * @param block Null indicates no block.
   */
  public abstract MethodT
    createMethod( TypeParameterT[] typeParameters,
									TypeReferenceT returnType,
									String name,
									FormalParameterListT parameters,
									ThrowsT throwsClause,
									BlockT block );

  /**
   * @param name Require non-null.
   * @param arguments Null indicates none.
   * @param body Null indicates none.
   */
  public abstract FieldDeclT
    createEnumConstant( String name,
                        ListExpressionT arguments,
                        ClassBodyT body );

  /**
   * Creates an instance class initializer.
   *
   * @param block Null indicates an empty block.
   */
  public final ClassInitializerT
    createInstanceInitializer( BlockT block )
  {
    return createClassInitializer( 0, block );
  }

  /**
   * Creates a static class initializer.
   *
   * @param block Null indicates an empty block.
   */
  public final ClassInitializerT
    createStaticInitializer( BlockT block )
  {
    return createClassInitializer( Modifier.STATIC, block );
  }

  /**
   * Creates a class initializer.
   *
   * @param modifiers A bit mask with values from java.lang.reflect.Modifier.
   * @param block Null indicates an empty block.
   */
  public abstract ClassInitializerT
    createClassInitializer( int modifiers, BlockT block );


  // ----------------------------------------------------------------------

  /**
   * Creates a field variable to match the local variable. Note, this
   * function does not convert the local variable. Instead, this
   * function creates a field that uses the variable's type and name.
   *
   * The input local variable should be discarded after this.
   *
   * @param variable Require non-null.
   */
  public abstract FieldVariableT
    createFieldVariable( LocalVariableT variable );

  /**
   * Creates a field variable to match the local variable. Note, this
   * function does not convert the local variable. Instead, this
   * function creates a field that uses the variable's type and name.
   *
   * The input local variable should be discarded after this.
   *
   * @param variable Require non-null.
   */
  public final FieldDeclT
    createFieldDeclaration( LocalVariableT variable )
  {
    final FieldVariableT fieldVariable =
      createFieldVariable( variable );
    return createFieldDeclaration( fieldVariable );
  }

  /**
   * Creates a field declaration.
   */
  public final FieldDeclT
    createFieldDeclaration( FieldVariableT variable )
  {
    return createFieldDeclaration( variable, FieldVariableT.EMPTY_ARRAY );
  }

  /**
   * Creates a field declaration.
   */
  public abstract FieldDeclT
    createFieldDeclaration( FieldVariableT one, FieldVariableT[] more );

  /**
   * Creates a field declaration to match the local variable
   * declaration.  Note, this function does not convert the local
   * variable declaration or any fo the variables therein. Instead,
   * this function creates a field variable for each local variable
   * and then creates a field declaration out of those field variables.
   *
   * The input local variable declaration should be discarded after
   * this.
   *
   * @param variables Require non-null.
   */
	public abstract FieldDeclT
		createFieldDeclaration( LocalVariableDeclT variables );


  // ----------------------------------------------------------------------

  /**
   * Creates a superclass clause for a class declaration.
   */
	public abstract SuperclassT
		createSuperclassClause( TypeReferenceT superclass );

  /**
   * Creates an interfaces clauses for a class declaration.
   */
	public final InterfacesT
		createInterfacesClause( TypeReferenceT oneInterface )
  {
    return createInterfacesClause( oneInterface, TypeReferenceT.EMPTY_ARRAY );
  }

  /**
   * Creates an interfaces clauses for a class declaration.
   */
	public abstract InterfacesT
		createInterfacesClause( TypeReferenceT one, TypeReferenceT[] more );

  /**
   * Creates a class body.
   */
  public final ClassBodyT createClassBody()
  {
    return createClassBody( MemberT.EMPTY_ARRAY );
  }

  /**
   * Creates a class body.
   */
  public final ClassBodyT createClassBody( MemberT member )
  {
    return createClassBody( new MemberT[] { member } );
  }

  /**
   * Creates a class body.
   */
  public abstract ClassBodyT createClassBody( MemberT[] members );


	// ----------------------------------------------------------------------

  /**
   * Creates a formal parameter list.
   */
  public final FormalParameterListT
    createFormalParameterList()
  {
    return createFormalParameterList( FormalParameterT.EMPTY_ARRAY );
  }

  /**
   * Creates a formal parameter list.
   */
  public final FormalParameterListT
    createFormalParameterList( FormalParameterT parameter )
  {
    return createFormalParameterList( new FormalParameterT[] { parameter } );
  }

  /**
   * Creates a formal parameter list.
   */
  public abstract FormalParameterListT
    createFormalParameterList( FormalParameterT[] parameters );

  /**
   * Creates a formal parameter.
   */
	public abstract FormalParameterT
		createFormalParameter( LocalVariableT variable );

  /**
   * Creates a throws clause for a method or constructor declaration.
   *
   * @return Null if empty.
   */
  public final ThrowsT createThrowsClause( TypeReferenceT exception )
  {
    return createThrowsClause( new TypeReferenceT[] { exception } );
  }

  /**
   * Creates a throws clause for a method or constructor declaration.
   *
   * @param exceptions Require non-null. For an empty exception array
   *                   use TypeReferenceT.EMPTY_ARRAY.
   *
   * @return Null if empty.
   */
  public abstract ThrowsT createThrowsClause( TypeReferenceT[] exceptions );


  // ----------------------------------------------------------------------

  /**
   * Creates a local variable
   *
   * @param type Require non-null.
   * @param name Require non-null.
   * @param initializer Null indicates none.
   */
  public abstract LocalVariableT
    createLocalVariable( TypeReferenceT type,
												 String name,
												 ExpressionT initializer );

  /**
   * Creates a local variable declaration.
   */
	public final LocalVariableDeclT
		createLocalVariableDeclaration( LocalVariableT variable )
  {
    return createLocalVariableDeclaration( variable,
                                           LocalVariableT.EMPTY_ARRAY );
  }

  /**
   * Creates a local variable declaration.
   */
	public abstract LocalVariableDeclT
		createLocalVariableDeclaration( LocalVariableT one,
                                    LocalVariableT[] more );


  // ----------------------------------------------------------------------

  /**
   * Creates an empty code block.
   */
  public final BlockT createBlock()
  {
    return createBlock( BlockElementT.EMPTY_ARRAY );
  }

  /**
   * Creates an empty code block.
   */
  public final BlockT createBlock( BlockElementT element )
  {
    return createBlock( new BlockElementT[] { element } );
  }

  /**
   * Creates a code block.
   *
   * @param elements The elements for the code block.
   */
  public abstract BlockT createBlock( BlockElementT[] elements );


  // ----------------------------------------------------------------------
  //  S T A T E M E N T S

  // ----------------------------------------------------------------------

  /**
   * Creates an assert statement.
   *
   * @param condition Require non-null.
   * @param output Require non-null.
   */
  public abstract AssertStatementT
    createAssertStatement( ExpressionT condition, ExpressionT output );

  /**
   * Creates a block statement.
   *
   * @param block Null indicates an empty block.
   */
  public abstract BlockStatementT
    createBlockStatement( BlockT block );

  /**
   * Creates a break statement.
   *
   * @param target Null indicates the implicitly enclosing loop.
   */
  public abstract BreakStatementT
    createBreakStatement( String target );

  /**
   * Creates a catch clause for a try-catch or a try-catch-finally
   * statement.
   *
   * The input parameter will be converted to be a formal parameter,
   * namely SRC_FORMAL_PARAMETER.
   *
   * @param parameter Require non-null.
   * @param child Null indicates an empty clause.
   */
  public final CatchClauseT
    createCatchClause( FormalParameterT parameter,
                       BlockStatementT child )
  {
    final FormalParameterListT parameters =
			createFormalParameterList( parameter );
    return createCatchClause( parameters, child );
  }

  /**
   * Creates a catch clause for a try-catch or a try-catch-finally
   * statement.
   *
   * @param parameters Require non-null. Must have exactly one parameter.
   * @param child Null indicates an empty clause.
   */
  public abstract CatchClauseT
    createCatchClause( FormalParameterListT parameters,
                       BlockStatementT child );

  /**
   * Creates a continue statement.
   *
   * @param target Null indicates the implicitly enclosing loop.
   */
  public abstract ContinueStatementT
    createContinueStatement( String target );

  /**
   * Creates a do-while statement.
   *
   * @param e Require non-null
   * @param child Null indicates an empty clause.
   */
  public abstract DoStatementT
    createDoStatement( ExpressionT e, StatementT child );

  /**
   * Creates an else clause for an if statement.
   *
   * @param child Null indicates an empty clause.
   */
  public abstract ElseClauseT
    createElseClause( StatementT child );

  /**
   * Creates an empty statement.
   */
  public abstract EmptyStatementT createEmptyStatement();

  /**
   * Creates an expression statement.
   *
   * @param e Require non-null.
   */
  public abstract ExpressionStatementT
    createExpressionStatement( ExpressionT e );

  /**
   * Creates a finally clause for a try statement.
   *
   * @param child Null indicates an empty clause.
   */
  public abstract FinallyClauseT
    createFinallyClause( BlockStatementT child );

  /**
   * Convenience method. Creates a traditional for loop with no
   * variable declarators and with at most one initialization and at
   * most one update.
   *
   * @param initialization Null means no initializations.
   * @param condition Null means an implicitly true condition.
   * @param update Null means no updates.
   * @param child Null indicates an empty clause.
   */
  public final ForStatementT
    createForStatement( ExpressionT initialization,
                        ExpressionT condition,
                        ExpressionT update,
                        StatementT child )
  {
    ListExpressionT inits = createExpressionList( initialization );
    ListExpressionT updates = createExpressionList( update );
      
    return createForStatement( inits, condition, updates, child );
  }

  /**
   * Creates a traditional for loop with no variable declarators.
   *
   * @param initializations Null means no initializations.
   * @param condition Null means an implicitly true condition.
   * @param updates Null means no updates.
   * @param child Null indicates an empty clause.
   */
  public abstract ForStatementT
    createForStatement( ListExpressionT initializations,
                        ExpressionT condition,
                        ListExpressionT updates,
                        StatementT child );

  /**
   * Convenience method. Creates a traditional for loop with a single
   * local variable and at most one update expression.
   *
   * @param variable Require non-null.
   * @param condition Null means an implicitly true condition.
   * @param update Null means no update.
   * @param child Null indicates an empty clause.
   */
  public final ForStatementT
    createForVariableStatement( LocalVariableT variable,
                                ExpressionT condition,
                                ExpressionT update,
                                StatementT child )
  {
    final LocalVariableDeclT variables =
      createLocalVariableDeclaration( variable );
    final ListExpressionT updates = createExpressionList( update );
    return createForVariableStatement( variables, condition, updates, child );
  }

  /**
   * Creates a traditional for loop with variable declarators.
   *
   * @param variables Require non-null. Require at least one variable.
   * @param condition Null means an implicitly true condition.
   * @param updates Null means no updates.
   * @param child Null indicates an empty clause.
   */
  public abstract ForStatementT
    createForVariableStatement( LocalVariableDeclT variables,
                                ExpressionT condition,
                                ListExpressionT updates,
                                StatementT child );

  /**
   * Creates an enhanced-for loop.
   *
   * @param loopVariable Require non-null.
   * @param collection Require non-null.
   * @param child Null indicates an empty clause.
   */
  public abstract ForStatementT
    createForEnhancedStatement( LocalVariableT loopVariable,
                                ExpressionT collection,
                                StatementT child );

  /**
   * Creates an if statement. Will not create the else clause.
   *
   * @param e Require non-null.
   * @param child Null indicates an empty clause.
   */
  public abstract IfStatementT
    createIfStatement( ExpressionT e,
											 StatementT child );

  /**
   * Creates an if or an if-else statement. Will not create the else
   * clause if none is given.
   *
   * @param e Require non-null
   * @param child Null indicates an empty true clause.
   * @param elseClause Null means no else clause.
   */
  public abstract IfStatementT
    createIfStatement( ExpressionT e,
                       StatementT child,
											 ElseClauseT elseClause );

  /**
   * Creates an if-else statement. Always creates the else clause, even
   * if empty.
   *
   * @param e Require non-null.
   * @param trueClause Null indicates an empty true clause.
   * @param elseClause Null indicates an empty else clause.
   */
  public final IfStatementT
    createIfElseStatement( ExpressionT e,
                           StatementT trueClause,
                           StatementT elseClause )
  {
    return createIfStatement( e, trueClause, createElseClause( elseClause ) );
  }

  /**
   * Creates a return statement.
   *
   * @param e Null indicates a void return.
   */
  public abstract ReturnStatementT
    createReturnStatement( ExpressionT e );

  /**
   * Creates a switch statement.
   *
   * @param e Require non-null.
   * @param child Null indicates an empty block.
   */
  public abstract SwitchStatementT
    createSwitchStatement( ExpressionT e,
                           BlockStatementT child );

  /**
   * Creates a synchronized statement.
   *
   * @param e Require non-null.
   * @param child Null indicates an empty block.
   */
  public abstract SynchStatementT
    createSynchronizedStatement( ExpressionT e,
                                 BlockStatementT child );

  /**
   * Creates a throw statement.
   *
   * @param e Require non-null.
   */
  public abstract ThrowStatementT
    createThrowStatement( ExpressionT e );

  /**
   * Creates a try, try-catch, try-catch-finally, or a try-finally
   * statement.
   *
   * @param tryClause Null indicates an empty try clause.
   * @param catchClause Require non-null. If you want no catch clauses,
   *                     use CatchClauseT.EMPTY_ARRAY.
   * @param finallyClause Null indicates no finally clause.
   */
  public final TryStatementT
    createTryStatement( BlockStatementT tryClause,
                        CatchClauseT catchClause,
                        FinallyClauseT finallyClause )
	{
		final CatchClauseT[] catchClauses = new CatchClauseT[] { catchClause };
		return createTryStatement( tryClause, catchClauses, finallyClause );
	}

  /**
   * Creates a try, try-catch, try-catch-finally, or a try-finally
   * statement.
   *
   * @param tryClause Null indicates an empty try clause.
   * @param catchClauses Require non-null. If you want no catch clauses,
   *                     use CatchClauseT.EMPTY_ARRAY.
   * @param finallyClause Null indicates no finally clause.
   */
  public abstract TryStatementT
    createTryStatement( BlockStatementT tryClause,
                        CatchClauseT[] catchClauses,
                        FinallyClauseT finallyClause );

  /**
   * Creates a try-catch statement with a single catch clause.
   *
   * @param tryClause Null indicates an empty try clause.
   * @param parameter Require non-null.
   * @param catchClause Null indicates an empty catch clause.
   */
  public abstract TryStatementT
    createTryCatchStatement( BlockStatementT tryClause,
                             FormalParameterT parameter,
                             BlockStatementT catchClause );

  /**
   * Creates a try-catch-finally statement with a single catch clause.
   * Always creates the finally clause.
   *
   * @param tryClause Null indicates an empty try clause.
   * @param parameter Require non-null.
   * @param catchClause Null indicates an empty catch clause.
   * @param finallyClause Null indicates an empty finally clause.
   */
  public abstract TryStatementT
    createTryCatchFinallyStatement( BlockStatementT tryClause,
                                    FormalParameterT parameter,
                                    BlockStatementT catchClause,
                                    BlockStatementT finallyClause );

  /**
   * Creates a try-finally statement. Always creates the finally clause.
   *
   * @param child Null indicates an empty try clause.
   * @param finallyClause Null indicates an empty finally clause.
   */
  public abstract TryStatementT
    createTryFinallyStatement( BlockStatementT child,
                               BlockStatementT finallyClause );

  /**
   * Creates a while statement.
   *
   * @param e Require non-null.
   * @param child Null indicates an empty clause.
   */
  public abstract WhileStatementT
    createWhileStatement( ExpressionT e, StatementT child );


  // ----------------------------------------------------------------------
  //  E X P R E S S I O N S

  // ----------------------------------------------------------------------

  /**
   * Creates an argument list.
   *
   */
  public final ListExpressionT
    createArgumentList()
  {
    return createArgumentList( ExpressionT.EMPTY_ARRAY );
  }

  /**
   * Creates an argument list.
   *
   * @param argument Null indicates an empty list.
   */
  public final ListExpressionT
    createArgumentList( ExpressionT argument )
  {
    return createArgumentList( new ExpressionT[] { argument } );
  }

  /**
   * Creates an argument list.
   *
   * @param arguments Null indicates an empty list.
   */
  public final ListExpressionT
    createArgumentList( ExpressionT[] arguments )
  {
    return createExpressionList( arguments );
  }

  /**
   * Creates an array access.
   *
   * @param lhs Require non-null.
   * @param dereferences Requires non-null. Use createArgumentList([).
   */
  public abstract ArrayAccessExpressionT
    createArrayAccess( ExpressionT lhs,
                       ListExpressionT dereferences );

  /**
   * Creates an array constant.
   */
  public final ListExpressionT
    createArrayConstant()
  {
    return createArrayConstant( ExpressionT.EMPTY_ARRAY );
  }

  /**
   * Creates an array constant.
   */
  public final ListExpressionT
    createArrayConstant( ExpressionT operand )
  {
    return createArrayConstant( new ExpressionT[] { operand } );
  }

  /**
   * Creates an array constant.
   */
  public abstract ListExpressionT
    createArrayConstant( ExpressionT[] operands );

  /**
   * Creates an array creator for an uninitialized array.
   *
   * @param type Require non-null.
   */
  public abstract NewArrayExpressionT
    createNewInitializedArrayExpression( TypeReferenceT type,
                                         ListExpressionT arrayconst );

  /**
   * Creates an array creator for an initialized array.
   */
  public abstract NewArrayExpressionT
    createNewUninitializedArrayExpression( TypeReferenceT type,
                                           ListExpressionT args );

  /**
   * Creates an assignment expression.
   */
  public abstract AssignmentExpressionT
    createAssignment( ExpressionKind kind, ExpressionT lhs, ExpressionT rhs );

  /**
   * Creates a class creator. Class creators have the form: <p/>
   *
   * <pre>
   *   new type( arg )
   *   new type( arg ) { void bodyMethods(); }
   *   lhs.new type( arg )
   *   lhs.new type( arg ) { void bodyMethods(); }
   * </pre>
   *
   * @param lhs Non-null indicates a qualified class creator (for
   *            creating non-static inner classes).
   * @param type Require non-null. Require non-array reference type.
   * @param arguments Null indicates no arguments. Use createArgumentList([).
   * @param body Non-null indicates an anonymous class.
   */
  public abstract NewClassExpressionT
    createNewClassExpression( ExpressionT lhs,
															TypeReferenceT type,
															ListExpressionT arguments,
															ClassBodyT body );

  /**
   * Creates a dot reference. Dot references have the form:
   *
   *   lhs.name
   *
   * @param lhs Require non-null.
   * @param name Require non-null.
   */
  public abstract DotExpressionT
    createDotReference( ExpressionT lhs, String name );

  /**
   * Creates a list expression. May not be used in place of an array
   * constant.
   */
  public final ListExpressionT
    createExpressionList()
  {
    return createExpressionList( ExpressionT.EMPTY_ARRAY );
  }

  /**
   * Creates a list expression. May not be used in place of an array
   * constant.
   */
  public final ListExpressionT
    createExpressionList( ExpressionT argument )
  {
    return createExpressionList( new ExpressionT[] { argument } );
  }

  /**
   * Creates a list expression. May not be used in place of an array
   * constant.
   */
  public abstract ListExpressionT
    createExpressionList( ExpressionT[] arguments );

  /**
   * Creates an infix expression with two operands.
   */
  public final InfixExpressionT
    createInfixExpression( ExpressionKind kind,
                           ExpressionT lhs, ExpressionT rhs )
  {
    return createInfixExpression( kind, lhs, rhs, ExpressionT.EMPTY_ARRAY );
  }

  /**
   * Creates an infix expression with multiple operands. An example
   * is: <p/>
   *
   * <pre>
   *   "Hello" + ' ' + "world!"
   * </pre>
   *
   * @param operands Must have at least two operands.
   */
  public abstract InfixExpressionT
    createInfixExpression( ExpressionKind kind, ExpressionT one,
                           ExpressionT two, ExpressionT[] operands );

  /**
   * Creates a new instanceof expression.
   *
   * @param lhs Require non-null.
   * @param type Require a reference type.
   */
  public abstract InfixExpressionT
    createInstanceof( ExpressionT lhs, TypeReferenceT type );

  /**
   * Creates a method call.
   *
   * @param lhs Null indicates the implicit "this" lhs.
   * @param name Require non-null.
   * @param arguments Null indicates no arguments. Use createArgumentList([).
   */
  public abstract MethodCallExpressionT
    createMethodCall( ExpressionT lhs,
											TypeArgumentT[] typeArguments,
											String name,
                      ListExpressionT arguments );

  /**
   * Creates a question expression.
   *
   * @param one The conditional operand. Require non-null.
   * @param two The operand executed by the true branch. Require non-null.
   * @param three The operand executed by the false branch. Require non-null.
   */
  public abstract QuestionExpressionT
    createQuestionExpression( ExpressionT one,
                              ExpressionT two,
                              ExpressionT three );

  /**
   * Creates a simple name expression.
   *
   * @param name Require non-null.
   */
  public abstract IdentifierExpressionT
    createIdentifierExpression( String name );

  /**
   * Creates a type expression.
   *
   * @param type Require non-null.
   */
  public abstract TypeExpressionT createTypeExpression( TypeReferenceT type );

  /**
   * Creates a typecast.
   *
   * @param type Require non-null. Require a reference type.
   * @param operand Require non-null.
   */
  public abstract TypecastExpressionT
    createTypecast( TypeReferenceT type, ExpressionT operand );

  /**
   * Creates a unary expression.
   *
   * @param operand Require non-null.
   */
  public abstract UnaryExpressionT
    createUnaryExpression( ExpressionKind kind, ExpressionT operand );

  /**
   * Creates a wrapper expression.
   *
   * @param operand Require non-null.
   */
  public abstract WrapperExpressionT
		createWrapperExpression( ExpressionT operand );



  // ----------------------------------------------------------------------

  /**
   * Creates a doc comment with the given raw text, with '\n' line
   * terminators.
   *
   * @param rawText Must be non-null.
   */
  public final DocCommentT createDocComment( String rawText )
  {
    return textToDocComment( rawText );
  }


  // ----------------------------------------------------------------------

}
