/*
 * @(#)Declaration.java
 */

package javax.ide.model.java.declaration;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import javax.ide.model.java.source.SourcePosition;

/**
 * Common supertype of all Declaration elements. A Declaration element
 * represents a declarative element in Java.
 *
 * @author Andy Yu
 */
public interface Declaration
{
  // ----------------------------------------------------------------------

  /**
   * Gets this Declaration's kind.
   *
   * @return This Declaration's kind.
   */
  public DeclarationKind getDeclarationKind();

  /**
   * Gets the position in a source compilation unit that declares that
   * declaration, null if none or if it could not be determined.
   *
   * @return Gets the position in a source compilation unit that
   * declares this declaration, null if none or if it could not be
   * determined.
   */
  public SourcePosition getPosition();


  // ----------------------------------------------------------------------

  /**
   * True if this is a synthetic element.
   *
   * @return True if this Declaration represents a synthetic element.
   */
  public boolean isSynthetic();


  // ----------------------------------------------------------------------

  /**
   * Identifies a Declaration.
   */
  public static final class DeclarationKind
  {
    private static final Map values = new LinkedHashMap();

    private static final int VALUE_ANNOTATION = 0;
    private static final int VALUE_ARRAY_TYPE = 1;
    private static final int VALUE_CLASS = 2;
    private static final int VALUE_CONSTRUCTOR = 3;
    private static final int VALUE_FIELD = 4;
    private static final int VALUE_LOCAL_VARIABLE = 5;
    private static final int VALUE_METHOD = 6;
    private static final int VALUE_PACKAGE = 7;
    private static final int VALUE_PARAMETER = 8;
    private static final int VALUE_TYPE_VARIABLE = 9;
    private static final int VALUE_WILDCARD_TYPE = 10;

    /** An annotation. */
    public static final DeclarationKind ANNOTATION =
      new DeclarationKind( VALUE_ANNOTATION, "ANNOTATION", AnnotationD.class );

    /** An array type. */
    public static final DeclarationKind ARRAY_TYPE =
      new DeclarationKind( VALUE_ARRAY_TYPE, "ARRAY_TYPE", ArrayTypeD.class );

    /** A class, interface, enum, or annotation type. */
    public static final DeclarationKind CLASS =
      new DeclarationKind( VALUE_CLASS, "CLASS", ClassD.class );

    /** A field or enum constant. */
    public static final DeclarationKind CONSTRUCTOR =
      new DeclarationKind( VALUE_CONSTRUCTOR, "CONSTRUCTOR",
                           ConstructorD.class );

    /** A field or enum constant. */
    public static final DeclarationKind FIELD =
      new DeclarationKind( VALUE_FIELD, "FIELD", FieldD.class );

    /** A method, constructor, or annotation element. */
    public static final DeclarationKind LOCAL_VARIABLE =
      new DeclarationKind( VALUE_LOCAL_VARIABLE, "LOCAL_VARIABLE",
                           LocalVariableD.class );
    
    /** A method, constructor, or annotation element. */
    public static final DeclarationKind METHOD =
      new DeclarationKind( VALUE_METHOD, "METHOD", MethodD.class );
    
    /** A method, constructor, or annotation element. */
    public static final DeclarationKind PACKAGE =
      new DeclarationKind( VALUE_PACKAGE, "PACKAGE", PackageD.class );
    
    /** A parameter. */
    public static final DeclarationKind PARAMETER =
      new DeclarationKind( VALUE_PARAMETER, "PARAMETER", ParameterD.class );

    /** A type variable. */
    public static final DeclarationKind TYPE_VARIABLE =
      new DeclarationKind( VALUE_TYPE_VARIABLE, "TYPE_VARIABLE", 
                           TypeVariableD.class );

    /** A wildcard type. */
    public static final DeclarationKind WILDCARD_TYPE =
      new DeclarationKind( VALUE_WILDCARD_TYPE, "WILDCARD_TYPE", 
                           WildcardTypeD.class );

    ;

    private final int ordinal;
    
    private final String name;

    private final Class exprClass;

    private DeclarationKind(int ordinal, String name, Class c)
    {
      this.ordinal = ordinal;
      this.name = name;
      this.exprClass = c;

      values.put( name, this );
    }

    public Class getExpressionClass()
    {
      return exprClass;
    }


    // ----------------------------------------------------------------------

    // Begin enum compatibility section.
    
    public String name()
    {
      return name;
    }
    
    public String toString()
    {
      return name();
    }
    
    public int ordinal()
    {
      return ordinal;
    }
    
    public int hashCode()
    {
      return ordinal();
    }
    
    public int compareTo(DeclarationKind other)
    {
      return ordinal() - other.ordinal();
    }
    
    public boolean equals(Object other)
    {
      if (other instanceof DeclarationKind)
      {
        final DeclarationKind tk = (DeclarationKind) other;
        return ordinal() == tk.ordinal();
      }
      
      return false;
    }

    public Class getDeclaringClass()
    {
      return DeclarationKind.class;
    }


    // ----------------------------------------------------------------------

    public static DeclarationKind valueOf(int ordinal)
    {
      return values()[ ordinal ];
    }

    public static DeclarationKind valueOf(Class ignored, String name)
    {
      return (DeclarationKind) values.get(name);
    }

    public static DeclarationKind[] values()
    {
      final Collection entries = values.values();
      return (DeclarationKind[])
        entries.toArray(new DeclarationKind[entries.size()]);
    }


    // ----------------------------------------------------------------------
  }
}
