/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.model;

import java.io.IOException;
import javax.swing.undo.UndoableEdit;

public interface Transaction
{
  /**
   * Used to instruct a model to start a compound edit
   * such that all the changes made between this point and when the
   * <code>endEdit()</code> is called should be combined into a single
   * UndoableEdit record.  This guarantees to the client that the
   * buffer will be locked for the duration between
   * <code>beginEdit()</code> and <code>endEdit()</code> for data
   * consistency.  This also takes care of locking the buffer, and
   * collecting the individual <code>UndoableEdit</code> objects into
   * a larger compound one. <p>
   *
   * Note that compound edits may <b>not</b> be nested - it is illegal
   * to call <code>beginEdit()</code> twice in a row without calling
   * <code>endEdit()</code> in between.
   * 
   * <pre>
   * Follow this pattern to make sure the edit is either committed
   * or rolled back correctly:
   *
   * beginEdit();
   * boolean editCommitted = false;
   * // No further code here
   *
   * try
   * {
   *   // Do the work
   *
   *   commitEdit();
   *   editCommitted = true;
   *
   *   // More work if necessary
   * }
   * catch( Exception ex)
   * {
   *   // Rethrow or log
   * }
   * finally
   * {
   *   // No other code here
   *   if (!editCommitted)
   *     rollbackEdit();
   *     
   *   // Other code
   * }
   * </pre>
   * 
   * @exception ReadOnlyException if the buffer is in read only mode
   */
  public void beginEdit() throws IOException;

  /**
   * Used to indicate to the text buffer to end an in progress
   * compound edit.  This will end the compound edit and returned the
   * single combined <code>UndoableEdit</code> representing all of the
   * changes made between the calls to <code>beginEdit()</code> and
   * <code>endEdit()</code>.  If no modifications were made to the
   * buffer since <code>beginEdit()</code> was called, null will be
   * returned.
   * @return the UndoableEdit representing all the changes made
   */
  public UndoableEdit commitEdit();

 /**
   * Used to indicate to the text buffer to end an in progress
   * compound edit without commiting the changes. The text buffer
   * will be left in the same state it was found when the call to
   * <code>beginEdit</code> was made.
   */
  public void rollbackEdit();

  /**
   * This allows the model to be safely updated.  The given runnable
   * will be executed in a way that allows it to safely write to the model
   * (with no changes from other threads) while the runnable is being executed.
   * Multiple write calls from the same thread are allowed.
   *
   * @param runnable  a <code>Runnable</code> used to write to the model.
   */
  public void write( Runnable runnable );

}

