/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.menu;

import javax.ide.Service;
import javax.ide.extension.ExtensionRegistry;
import javax.ide.menu.spi.MenuHook;
import javax.ide.menu.spi.MenuModel;
import javax.ide.spi.ProviderNotFoundException;

/**
 * Action lookup service. Extensions use this class to lookup 
 * {@link IDEAction}s.
 */
public class ActionRegistry extends Service 
{
  /**
   * Find the {@link IDEAction} identified by the given <code>id</code>.<p>
   * 
   * This implementation retrieves the action from the menu model and returns
   * it.
   *
   * @param id A unique string name identifying the action. Standard 
   * action identifiers are defined in the class {javax.ide.IDEConstants}.
   * @return An existing action if one is found, <code>null</code> otherwise.
   */
  public IDEAction findAction( String id )
  {
    return (IDEAction) getModel().getActions().get( id );
  }
  
  /**
   * Initialize the action registry.<p>
   * 
   * This implementation does nothing. IDE implementations may override this
   * method to do additional work when the action registry is initialized.
   */
  protected void initialize()
  {

  }

  /**
   * Convenience method that obtains the menu model from the MenuHook.
   * 
   * @return the menu model.
   */
  protected final MenuModel getModel()
  {
    MenuHook menuHook = (MenuHook) 
      ExtensionRegistry.getExtensionRegistry().getHook(
        MenuHook.ELEMENT );
    return menuHook.getModel();
  }
  
  /**
   * Get the ActionRegistry implementation for this IDE.
   * 
   * @return the ActionRegistry implementation for this IDE.
   */
  public static ActionRegistry getActionRegistry()
  {
    try
    {
      return (ActionRegistry) getService( ActionRegistry.class  );
    }
    catch ( ProviderNotFoundException nse )
    {
      nse.printStackTrace();
      throw new IllegalStateException( "No action registry" );
    }
  }
  
  public void initAfterMainWindowOpened()
  {
    // no op
  }
  
  public void initAfterAddinsLoad()
  {
    // no op
  }
}
