/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension.spi;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import javax.ide.extension.Extension;

/**
 * Represents an extension source. An extension source is typically a JAR file
 * in an IDE specific location for extension jar files. However, an 
 * ExtensionSource could potentially be some other abstract representation of
 * an extension. One example would be an as-yet unbundled extension manifest
 * launched from the extension development runtime of an IDE.<p>
 * 
 * A default implementation of this interface is provided in JARExtensionSource.
 * 
 */
public interface ExtensionSource 
{
  /**
   * Get the classpath entry for this extension source. This is the
   * (optional) entry that should be added to the classpath prior to 
   * processing this extension.
   * 
   * @return an {@link URI} to the classpath to be used by the class loader
   * that will load an extension classes.
   */
  public URI getClasspathEntry();

  /**
   * Resolve a (relative) path referenced in the manifest file for this 
   * extension source. 
   * 
   * @param extension the extension being processed.
   * @param path a relative path used in the manifest file of this extension
   *    source.
   * @return an absolute URI pointing to the location of the specified 
   *    resource.
   */
  public URI resolvePath( Extension extension, String path );
  
  /**
   * Get an input stream for this sources manifest.
   * 
   * @return an input stream for this sources manifest.
   */
  public InputStream getInputStream() throws IOException;
  
  /**
   * Get the URI of the manifest file contained in this source.  
   * 
   * @return the URI of the manifest file. May return null, in which case
   *    this source will be skipped.
   */
  public URI getManifestURI();
  
  /**
   * Get the name of this source. This will be used to present the source
   * in human readable messages.
   * 
   * @return the name of this source.
   */
  public String getName();
  
  /**
   * Get the URI of this source. This is the container of the extension.
   * 
   * @return the URI of the source. For JAR extensions, this is the URI of
   *  the jar file.
   */
  public URI getURI();
  
}
