package javax.ide.extension.spi;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementEndContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.ElementVisitorFactory;

import javax.xml.stream.Location;


/**
 * Helper class that should only be called by DeferredElementVisitorHook.
 * This class provides the implementation for recording XML data passed
 * to ElementVisitor methods and replaying them to a different
 * ElementVisitor.
 */
public class DeferredElementVisitorHelper
{
  public DeferredElementVisitorHelper()
  {
    super();
  }
  
  /**
   * Record the XML data associated with the given top-level start element (and its
   * children).
   * @param context
   */
  public void recordTopLevelElementStart(ElementStartContext context)
  { 
    TopLevelElementData elementData =  new TopLevelElementData(context.getElementName(),
                                                               _getCurrentLocator(context),
                                                               context.getScopeData());
    
    _recordAttributes(context, elementData);
    _topLevelElements.add(elementData);
    _setCurrentParent(context, elementData);
    context.registerVisitorFactory(_recordElementDataVisitorFactory);
  }

  /**
   * Record the XML data associated with the given top-level end element
   * @param context
   */
  public void recordTopLevelElementEnd(ElementEndContext context)
  {
    ElementData elementData = _getCurrentParent(context);
    _recordText(context, elementData);
  }

  
  /**
   * Replay all the recorded XML data into the given ElementVisitor's
   * start/end methods.  
   * @param context
   * @param visitor
   */
  public void visitRecordedData(DefaultElementContext context, 
                                ElementVisitor visitor)
  {
    if (visitor == null)
    {
      throw new NullPointerException("visitor == null");
    }
    
    int size = _topLevelElements.size();
    for (int i=0; i<size; i++)
    {
      TopLevelElementData element = _topLevelElements.get(i);
      context.getScopeData().putAll(element.getRecordedScopeData());
      _visitElement(visitor, element, context);
      context.reset();
    }
    _visitorToAlreadySeenIndex.put(visitor, size);
    
  }

  /**
   * Can be called multiple times with the same visitor, and it will
   * only replay any newly recorded XML data that this visitor
   * instance hasn't already seen
   * @param context
   * @param visitor
   */
  public void visitNewlyRecordedData(DefaultElementContext context, 
                                     ElementVisitor visitor)
  {
    if (visitor == null)
    {
      throw new NullPointerException("visitor == null");
    }
    
    Integer alreadySeen = _visitorToAlreadySeenIndex.get(visitor);
    if (alreadySeen == null)
    {
      alreadySeen = 0;
    }
    
    int size = _topLevelElements.size();
    for (int i=alreadySeen; i<size; i++)
    {
      TopLevelElementData element = _topLevelElements.get(i);
      context.getScopeData().putAll(element.getRecordedScopeData());
      _visitElement(visitor, element, context);
      context.reset();
    }
    _visitorToAlreadySeenIndex.put(visitor, size); 
  }

  
  /**
   * Clear the recorded XML data
   */
  public void clearRecordedData()
  {
    _topLevelElements.clear();
    _visitorToAlreadySeenIndex.clear();
  }
  
  
  private void _visitElement(ElementVisitor visitor, ElementData element, DefaultElementContext context)
  {
    //This implementation tries to replicate the pattern of calls in 
    //non-deferred extension.xml processing (see SaxManifiestParser).
    
    //In short, it sets up the context object for the element, calls the start method,
    //recursively processes the children, processes the text (if any), and calls
    //the end method.
    
    ElementName name = element.getName();
    context.beginElement(name.getNamespaceURI(), 
                         name.getLocalName(), 
                         element.getAttributes());
    
    _setCurrentLocator(context, element.getRecordedLocator());

    if (visitor != null)
    {
      try
      {
        visitor.start(context);
      }
      catch (Throwable re)
      {
        if (re instanceof ThreadDeath) throw (ThreadDeath) re;
        LogRecord r = new ExtensionLogRecord(context,
                                            Level.SEVERE,
                                            "Exception processing manifest: " +
                                            re.getClass().getName() + ":" + re.getMessage());
        r.setThrown(re);
        context.getLogger().log(r);
      }
    }

    context.postBeginElement();
    
    for (ElementData child : element.getChildren())
    {
      ElementVisitor childVisitor = context.getVisitorForStartElement(child.getName());
      _visitElement(childVisitor, child, context);
    }
    
    String text = element.getText();
    if ((text != null) && (text.length() > 0))
    {
      char[] chars = text.toCharArray();
      context.appendCharacters(chars, 0, chars.length);  
    }
    
    context.endElement(name.getNamespaceURI(),
                       name.getLocalName());

    if (visitor != null)
    {
      try
      {
        visitor.end(context);
      }
      catch (Throwable re)
      {
        if (re instanceof ThreadDeath) throw (ThreadDeath) re;
        LogRecord r = new ExtensionLogRecord(context,
                                             Level.SEVERE, "Exception processing manifest: " +
                                             re.getClass().getName() + ":" + re.getMessage());
        r.setThrown(re);
        context.getLogger().log(r);
      }
    }

    context.postEndElement();
  }
  

  private void _setCurrentParent(ElementContext context, ElementData elementData)
  {
    context.getScopeData().put(_CURRENT_PARENT, elementData);
  }
  
  private ElementData _getCurrentParent(ElementContext context)
  {
    return (ElementData) context.getScopeData().get(_CURRENT_PARENT);    
  }  

  private void _setCurrentLocator(ElementContext context, LocationAdapter locator)
  {
    context.getScopeData().put(ElementVisitor.KEY_LOCATOR, locator);    
  }
  
  private LocationAdapter _getCurrentLocator(ElementContext context)
  {
    return (LocationAdapter) context.getScopeData().get(ElementVisitor.KEY_LOCATOR);
  }
  
  private void _recordAttributes(ElementStartContext context, ElementData elementData)
  {
    if (context instanceof DefaultElementContext)
    {
      //Optimization to get the raw unprocessed form of the data (no need to go 
      //through the resource bundle lookups here, that will happen later when
      //the attached visitor is fed the data)
      DefaultElementContext.Attributes attributes = ((DefaultElementContext) context).getRawAttributes();
      if (attributes != null)
      {
        elementData.getAttributes().setAttributes(attributes);
      }
    }
    else
    {
      Collection<String> attrNames = context.getAttributeNames();
      for (String attrName : attrNames)
      {
        elementData.addAttribute(attrName, context.getAttributeValue(attrName));
      }
    }    
  }
  
  private void _recordText(ElementEndContext context, ElementData elementData)
  {
    if (context instanceof DefaultElementContext)
    {
      //Optimization to get the raw unprocessed form of the data (no need to go 
      //through the resource bundle lookups here, that will happen later when
      //the attached visitor is fed the data)    
      elementData.setText(((DefaultElementContext) context).getRawText());
    }
    else
    {
      elementData.setText(context.getText());
    }   
  }  
  

  private class RecordElementDataVisitorFactory implements ElementVisitorFactory
  { 
    public ElementVisitor getVisitor(ElementName name)
    {
      return _recordElementDataVisitor;
    }
  }
  
  private class RecordElementDataVisitor extends ElementVisitor
  {
    
    @Override
    public void start(ElementStartContext context)
    {
      ElementData child = new ElementData(context.getElementName(),
                                          _getCurrentLocator(context));
      _recordAttributes(context, child);
      ElementData parent = _getCurrentParent(context);
      parent.addChild(child);
      
      _setCurrentParent(context, child);
      context.registerVisitorFactory(_recordElementDataVisitorFactory);
    }

    @Override
    public void end(ElementEndContext context)
    {
      ElementData elementData = _getCurrentParent(context);
      _recordText(context, elementData);
    }    
  }
    
  private class ElementData
  {
    public ElementData(ElementName name,
                       LocationAdapter locator)
    {
      _name = name;
      _recordedLocator = XMLParsingUtils.copyAndCastToLocationAdapter(locator);
    }
    
    public ElementName getName()
    {
      return _name;
    }
    
    public void addAttribute(String attrName,
                             String value)
    {
      //JRS -- ElementContext APIs only expose the local names of attributes, so
      //that is all we have.  DefaultElementContext expects a SAX Attributes
      //object even though it will only use/expose the local name and value.
      //So, we pass null for the namespace here.  If ElementContext is changed
      //to expose a qualified name, we'll need to change this code too.
      _attributes.addAttribute(new PullManifestParser.StAXAttribute( attrName, value));
    }
    
    public DefaultElementContext.AttributesImpl getAttributes()
    {
      return _attributes;
    }
        
    public void setText(String text)
    {
      _text = text;
    }
    
    public String getText()
    {
      return _text;
    }
    
    public List<ElementData> getChildren()
    {
      if (_children == null)
      {
        return Collections.emptyList();
      }
      
      return _children;
    }
    
    public void addChild(ElementData child)
    {
      if (_children == null)
      {
        _children = new ArrayList<ElementData>(5);
      }
      
      _children.add(child);
    }
    
    private LocationAdapter getRecordedLocator()
    {
      return _recordedLocator;
    }
    
    private final ElementName _name;
    private DefaultElementContext.AttributesImpl _attributes = new DefaultElementContext.AttributesImpl();
    private String _text = null;
    private List<ElementData> _children;
    private final LocationAdapter _recordedLocator;
  }
  
  private class TopLevelElementData extends ElementData
  {
    
    public TopLevelElementData(ElementName name,
                               LocationAdapter locator,
                               Map scopeData)
    {
      super(name, locator);      
      //Copy the scope data, and clear the locator key so that we don't
      //hold on to the old locator (it is not necessary since we
      //already hold a locator snapsot, and holding it pins SAX 
      //resources like the XMLReader -- see bug 11939231)
      _recordedScopeData = ScopedMap.copyScopeData(scopeData,
                                                   Collections.singleton(ElementVisitor.KEY_LOCATOR));
    }
    
    
    private Map getRecordedScopeData()
    {
      return _recordedScopeData;
    }
    
    private final Map _recordedScopeData;
  }
  
  private List<TopLevelElementData> _topLevelElements = new CopyOnWriteArrayList<TopLevelElementData>();
  private ElementVisitorFactory _recordElementDataVisitorFactory = new RecordElementDataVisitorFactory();
  private RecordElementDataVisitor _recordElementDataVisitor = new RecordElementDataVisitor();
  private static final String _CURRENT_PARENT = "deferred-element-visitor-current-parent";
  private final WeakHashMap<ElementVisitor, Integer> _visitorToAlreadySeenIndex = new WeakHashMap<ElementVisitor, Integer>();
}
