/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension;

import java.net.URI;
import java.util.Map;
import java.util.logging.Logger;

/**
 * ElementContext provides information about the current element being visited
 * and provides utilities for processing child elements, reporting logging
 * messages and storing information for use by other visitors.
 */
public interface ElementContext 
{
  /**
   * Registers a visitor that will visit specific direct children of the current
   * element.
   * 
   * @param name the qualified name of child elements to visit.
   * @param visitor the visitor to register.
   */
  void registerChildVisitor( ElementName name, ElementVisitor visitor );
    
  /**
   * Registers a factory that can dynamically retrieve visitors for named
   * elements.
   * 
   * @param factory a factory used to create visitors for specific elements.
   */
  void registerVisitorFactory( ElementVisitorFactory factory );

  /**
   * Gets the name of the current element.
   * 
   * @return the name of the current element.
   */
  ElementName getElementName();
  
  /**
   * Gets scope data for the current context. The returned map is 
   * <b>scoped</b>: values {@link java.util.Map#put( Object, Object ) put}
   * into the map can only be retrieved by the current visitor or visitors
   * of descendent elements of the current element.
   * 
   * @return a scoped map.
   */
  Map getScopeData();

  /**
   * Gets the logger to which warning, error and information messages should be 
   * sent by the visitor.
   * 
   * @return an object used to report messages about processing of xml.
   */
  Logger getLogger();
  
  /**
   * Gets the extension currently being processed.
   * 
   * @return the extension currently being processed.
   */
  Extension getExtension();
  
  /**
   * Gets the URI of the source of the extension. For extensions packaged in
   * JAR files, this is the URI of the JAR file.
   * 
   * @return the URI of the exension source.
   */
  URI getExtensionSourceURI();
  
}
