/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.ide.extension.spi.DefaultElementContext;
import javax.ide.extension.spi.Stack;
import javax.ide.util.MetaClass;


/**
 * An implementation of <tt>ExtensionHook</tt> that automatically populates
 * model objects using reflection.
 * <p>
 * This class now strongly encourages delayed classloading through the
 * definition of {@link MetaClass} handling set/add methods on model
 * classes.
 *
 * In future, DynamicHook will be moved over to a strict classloading
 * policy requiring that any application objects whose classes are not
 * accessible to the classloader of the hook handler implementation must
 * be set on the model through new <tt>MetaClass</tt> signature
 * methods.
 * <p>
 * Currently, the Boolean system property <tt>DynamicHook.strict</tt> is
 * provided to allow implementors to ensure that their handlers and models
 * operate correctly under strict classloading. Note that this policy will
 * soon become the default behavior, and later the only supported behavior
 * of DynamicHook. The use of this option is recommended in the interests
 * of determining future compliance. In strict mode, any failures will be
 * reported to the {@link Logger} of the {@link ElementContext}.
 */
public class DynamicHook extends ExtensionHook
{
  public final String sApplicationObjectKey =
    DynamicHook.class.getName() + ".appObjectKey";

  /**
   * Strict mode constant; this mode enforces that all runtime types must be 
   * accessible by the classloader of the hook handler.
   */
  private static final boolean STRICT = Boolean.getBoolean("DynamicHook.strict");

  private static final String ATTRIBUTE_CLASS = "class";
  private static final String SET_METHOD_PREFIX = "set";
  private static final String ADD_METHOD_PREFIX = "add";
  private static final Class[] sObjectParamTypes = { Object.class };
  private static final Class[] sStringParamTypes = { String.class };
  private static final Class[] sClassParamTypes = { MetaClass.class };

  /**
   * This stack references the application objects currently in scope. After
   * creation, an application object is pushed onto the stack. When that
   * complex element goes out of scope, the object is popped from the stack.
   */
  private final Stack<StackObject> _applicationObjectStack = new Stack<StackObject>();
  
  /**
   * This stack references a verdict on whether the element in scope is complex
   * or simple. When the element goes out of scope, the verdict on the parent
   * element is available at the top of this stack.
   */
  private final Stack<boolean[]> _complexTypeIndicatorStack = new Stack<boolean[]>();

  /**
   * If non-null, this classloader is passed to all calls to Class.newInstance()
   * , otherwise the current thread context classloader is used.
   */
  private ClassLoader _classLoader;

  /**
   * A list of registered {@link ElementTypeResolver}s which may be queried for
   * the runtime type corresponding to an xml element.
   */
  private final List<ElementTypeResolver> _resolvers = new ArrayList<ElementTypeResolver>(5);
  
  private final ReentrantLock _handlerLock = new ReentrantLock();  
  private DeferredContext _deferredContext = null;

  public DynamicHook(Object rootObject)
  {
    // Push the root object into first place on the stack.
    _applicationObjectStack.push(new StackObject(rootObject));
  }
  
  public DynamicHook(Object rootObject, ClassLoader classLoader)
  {
    this(rootObject);
    _classLoader = classLoader;
  }

  public DynamicHook(
    Object rootObject,
    ClassLoader classLoader,
    ElementTypeResolver resolver)
  {
    this(rootObject, classLoader);
    _resolvers.add(resolver);
  }

  public void registerElementTypeResolver(ElementTypeResolver resolver)
  {
    _resolvers.add(resolver);
  }
  
  /**
   * If the hook implementation and its associated model use delayed 
   * classloading (as recommended), any {@link ElementVisitorFactory}
   * being attached to a handled context must created by this method. 
   * That means either; overriding this method to return the visitor 
   * factory instance, or; enabling this method to create a default
   * visitor factory by overriding {@link #getHookNamespaceURI} to 
   * return a non-null namespace value that applies to all handled 
   * elements.
   * @return the {@ ElementVisitorFactory} to attach to the context.
   */  
  protected ElementVisitorFactory createVisitorFactory()
  {
    final String hookNamespaceURI = getHookNamespaceURI();
    if (hookNamespaceURI == null)
      return null;
    
    return new ElementVisitorFactory()
      {
        public ElementVisitor getVisitor(
          ElementName name )
        {
          String namespaceURI = name.getNamespaceURI();
          if (namespaceURI != null && namespaceURI.equals(hookNamespaceURI))
          {
            return DynamicHook.this;
          }
          return null;
        }
      };
  }
  
  protected String getHookNamespaceURI()
  {
    return null;
  }
  
  private final boolean isProcessingDeferredTopLevel()
  {
    return (_deferredContext != null && peekParent() == _deferredContext.getDynamicMetaClass().getParent());    
  }
  
  @Override
  public void start(ElementStartContext context)
  {    
    _handlerLock.lock();

    if ((_deferredContext == null && _complexTypeIndicatorStack.isEmpty()) || isProcessingDeferredTopLevel())
    {
      ElementVisitorFactory visitorFactory = createVisitorFactory();
      if (visitorFactory != null)
        context.registerVisitorFactory(visitorFactory);      
    }
    
    MetaClass strictRuntimeMetaType = getRuntimeMetaType(context, true);
    _complexTypeIndicatorStack.push(new boolean[]
      {
        strictRuntimeMetaType != null ? true : false
      });
    if (strictRuntimeMetaType != null)
    {
      // The preferred technique for DynamicHook implementations is now to 
      // provide MetaClass handling methods in the model. However we must 
      // consider backward compatibility. So in order for us to instantiate 
      // the runtime type and set the instance on the model, the following 
      // conditions need to be met:
      // - if we are running with the STRICT system property set:
      //   - the runtime type being set must be accessible from the classloader
      //     of the hook handler implementation, and
      //   - there must exist a method on the parent application object matching
      //     the signature setXXX(Object) or addXXX(Object).
      // - if we are running without the STRICT system property set:
      //   - there must *not* exist a method on the parent application object 
      //     matching the signature setXXX(MetaClass) or addXXX(MetaClass).    
      boolean instantiate = true;  
      if (!isProcessingDeferredTopLevel())
      {
        if (STRICT)
        {
          try
          {
            strictRuntimeMetaType.toClass();
            
            findMethodEx(peekParent(), context.getElementName().getLocalName(), 
              sObjectParamTypes);
          }
          catch (ClassNotFoundException cnfe)
          {
            instantiate = false;
          }
          catch (NoSuchMethodException nsme)
          {
            instantiate = false;
          }
        }
        else
        {
          try
          {
            findMethodEx(peekParent(), context.getElementName().getLocalName(), 
              sClassParamTypes);
            instantiate = false;
          }
          catch (NoSuchMethodException nsme)
          {
            ;
          }           
        }
      }

      if (instantiate)      
      {
        Object o = null;
        {
          Exception e = null;
          try
          {
            // Retrieve the application object for the complex element.
            o = getApplicationObject(getRuntimeMetaType(context), context);              
  
            if (_deferredContext != null && isProcessingDeferredTopLevel())
              _deferredContext.setApplicationObject(o);
          }
          catch (InstantiationException ie)
          {
            if (_deferredContext != null)
              _deferredContext.setInstantiationException(ie);
            e = ie;
          }
          catch (IllegalAccessException iae)
          {
            if (_deferredContext != null)
              _deferredContext.setIllegalAccessException(iae);
            e = iae;
          }
          catch (ClassNotFoundException cnfe)
          {
            if (_deferredContext != null)
              _deferredContext.setClassNotFoundException(cnfe);
            e = cnfe;
          }  
          catch (Exception ex)
          {
            e = ex;
          }
          if (e != null)
          {
            _applicationObjectStack.push(new StackObject((Object)null));
            return;
          }
        }
        
        // Pre initialization hook.
        invokePreInitialize(peekParent(), o);

        // Push the application object onto the stack so that it becomes the
        // object in scope for child elements.
        _applicationObjectStack.push(new StackObject(o));

        // Push the application object onto the context map in scope. This makes
        // it available globally under the sApplicationObjectKey key. Note:
        // keeping a separate stack of app objects maintains the integrity of the
        // dynamic handler instance, since child handlers are at liberty to remove
        // an object from the context map.
        context.getScopeData().put(sApplicationObjectKey, o);
        
        // Additional complex element start behaviour, e.g. specialized attribute
        // handling.
        handleComplexElementStart(o, context);
      }
      else
      {
        DynamicMetaClass clazz = new DynamicMetaClass(getRuntimeMetaType(context), 
                                                      _applicationObjectStack.peek().getObject());                  
        
        clazz.getDeferredHook().start(context);
        
        // Push the application object onto the stack so that it becomes the
        // object in scope for child elements.
        _applicationObjectStack.push(new StackObject(clazz));   
      }
    }
    else
    {
      // Additional simple element start behaviour, e.g. specialized attribute
      // handling.
      handleSimpleElementStart(context);
    }
  } 
  
  @Override
  public void end(ElementEndContext context)
  {
    try
    {
      boolean isComplex = ((boolean[])_complexTypeIndicatorStack.pop())[0];
      if (isComplex)
      {
        // Reference the child object.
        StackObject stackObject = _applicationObjectStack.pop();
        Object childObject = stackObject.getObject();
        DynamicMetaClass childClass = stackObject.getDynamicMetaClass();
  
        if (childObject == null && childClass == null)
        {
          log(context, Level.SEVERE, "No application object or MetaClass");
          return;
        }
          
        Object parent = peekParent();
        if (parent == null)
        {
          log(context, Level.SEVERE, "No parent application object");
          return;
        }
        
        if (childObject != null)
        {
          // Post initialization hook.
          invokePostInitialize(childObject);
    
          // Attach the child and parent objects.
          if (!isProcessingDeferredTopLevel())
            attachObject(parent, childObject, context);
            
          // Hook for additional complex element end behaviour (none by default).
          handleComplexElementEnd(context);
        }
        else
        {
          childClass.getDeferredHook().end(context);
          
          if (!isProcessingDeferredTopLevel())
            attachClass(parent, childClass, context);        
        }
      }
      else
      {
        // Attach any simple element data to the application object in scope.
        Object parent = peekParent();      
        if (parent != null)
          attachData(parent, context.getText(), context);
  
        // Hook for additional simple element end behaviour (none by default).
        handleSimpleElementEnd(context);
      }
    }
    finally
    {
      _handlerLock.unlock();
    }     
  }
  
  private final Object peekParent()
  {
    return _applicationObjectStack.peek().getObject();
  }
  
  protected void handleComplexElementStart(
    Object applicationObject,
    ElementStartContext context)
  {
    // This basic implementation only knows about the 'class' attribute for
    // complex elements. However, specializations may override at this point
    // to plug in additional behaviour.
  }
  
  protected void handleComplexElementEnd(ElementEndContext context)
  {
    // This implementation takes no more action on ending of a complex element
    // after the child object has been attached to the parent object.
  }

  protected void handleSimpleElementStart(ElementStartContext context)
  {
    // This basic implementation ignores all attributes for simple elements.
    // However, specializations may override at this point to plug in additional
    // behaviour.
  }

  protected void handleSimpleElementEnd(ElementEndContext context)
  {
    // This implementation takes no more action on ending of a simple element
    // after the child data has been attached to the object in scope.
  }

  protected <T> MetaClass<T> getRuntimeMetaType(ElementStartContext context)
  {
    return getRuntimeMetaType(context, false);
  }
  
  private final <T> MetaClass<T> getRuntimeMetaType(ElementStartContext context, boolean strict)
  {
    Class type = getRuntimeType(context);
    String className = (type != null? type.getName(): context.getAttributeValue(ATTRIBUTE_CLASS));
    return (className != null? new MetaClass<T>(getHandlerClassLoader(context, strict), className): null);
  }
 
  /**
   * @deprecated replaced by {@link #getRuntimeMetaType(ElementStartContext)}
   */
  @Deprecated
  protected Class getRuntimeType(ElementStartContext context)
  {
    // First, interrogate the 'class' attribute. Failing that, interrogate the
    // registered resolvers in reverse order.
    Class type = null;
    String classAttribute = context.getAttributeValue(ATTRIBUTE_CLASS);
    if (classAttribute != null)
    {
      try
      {
        type = Class.forName(classAttribute, true, getHandlerClassLoader(context));
      }
      catch (ClassNotFoundException cnfe)
      {
        // The runtime type is not reachable from the hook handler classloader.
        // That's OK, we may be running in strict mode and a valid MetaClass 
        // could still be constructed based on the class attribute value and 
        // the context extension classloader.
        return null;
      }
    }
    else
    {
      ListIterator it = _resolvers.listIterator(_resolvers.size());
      while (it.hasPrevious() && type == null)
      {
        type = ((ElementTypeResolver)it.previous()).resolveType(
          context.getElementName() );
      }
    }
    return type;
  }
  
  private final ClassLoader getHandlerClassLoader(ElementContext context)
  {
    return getHandlerClassLoader(context, STRICT);
  }
    
  private final ClassLoader getHandlerClassLoader(ElementContext context, boolean strict)
  {
    if (_classLoader != null)
      return _classLoader;
    
    return (strict? getClass().getClassLoader(): getMetaClassLoader(context));
  }
    
  protected <T> T getApplicationObject(MetaClass<T> runtimeMetaType, ElementStartContext context)
    throws InstantiationException, IllegalAccessException, ClassNotFoundException  
  {
    try
    {
      return runtimeMetaType.newInstance();
    }
    catch (InstantiationException ie)
    {
      log( context, Level.SEVERE, "Unable to instantiate class: " + runtimeMetaType.getClassName() );       
      throw ie;
    }
    catch (IllegalAccessException iae)
    {
      log( context, Level.SEVERE, "Unable to access class or constructor: " + runtimeMetaType.getClassName() );
      throw iae;
    }
    catch (ClassNotFoundException cnfe)
    {
      log( context, Level.SEVERE, "Unable to load class: " + runtimeMetaType.getClassName() );       
      throw cnfe;
    }
  }
  
  /**
   * @deprecated replaced by {@link #getApplicationObject(MetaClass,ElementStartContext)}
   */  
  @Deprecated
  protected Object getApplicationObject(
    Class runtimeType,
    ElementStartContext context)
  {
    // In this simple implementation, the only information needed to retrieve
    // the application object is its type, however extra context is provided
    // in this method for the benefit of specializations.
    try
    {
      return runtimeType.newInstance();
    }
    catch (Exception e)
    {
      log( context, Level.SEVERE, "Unable to instantiate class: " +
        runtimeType.getName() );
      e.printStackTrace();
      return null;
    }
  }
  
  protected void attachClass(
    Object parent,
    MetaClass metaClass,
    ElementEndContext context)
  {
    // Find the relevant setXXX(MetaClass) or addXXX(MetaClass) method on the
    // parent application object.
    Method m = findMethod(context, parent, context.getElementName().getLocalName(), 
      sClassParamTypes);
    
    if (m == null)
      return;    
    
    // Dynamically invoke the target method, passing the child object as the
    // parameter to the target method.
    try
    {
      m.invoke(parent, new Object[] {metaClass});
    }
    catch (InvocationTargetException ite)
    {
      StringBuffer b = new StringBuffer(200);
      b.append("Could not attach class: ").append(metaClass.toString());
      b.append(" to parent: ").append(parent.toString());
      b.append(". Root cause: ");
      b.append(ite.getTargetException().getClass().getName()).append(": ");
      b.append(ite.getTargetException().getMessage());
      log( context, Level.SEVERE, b.toString() );
    }
    catch (Exception e)
    {
      StringBuffer b = new StringBuffer(200);
      b.append("Could not attach class: ").append(metaClass.toString());
      b.append(" to parent: ").append(parent.toString());
      b.append(". Root cause: ");
      b.append(e.getClass().getName()).append(": ").append(e.getMessage());
      log( context, Level.SEVERE, b.toString() );
    }    
  }
  
  protected void attachObject(
    Object parent,
    Object child,
    ElementEndContext context)
  {
    // Find the relevant setXXX(Object) or addXXX(Object) method on the
    // parent application object.
    Method m = findMethod(context, parent, context.getElementName().getLocalName(), 
      sObjectParamTypes);
    
    if (m == null)
      return;    
    
    // Dynamically invoke the target method, passing the child object as the
    // parameter to the target method.
    try
    {
      m.invoke(parent, new Object[] {child});
    }
    catch (InvocationTargetException ite)
    {
      StringBuffer b = new StringBuffer(200);
      b.append("Could not attach child object: ").append(child.toString());
      b.append(" to parent: ").append(parent.toString());
      b.append(". Root cause: ");
      b.append(ite.getTargetException().getClass().getName()).append(": ");
      b.append(ite.getTargetException().getMessage());
      log( context, Level.SEVERE, b.toString() );
    }
    catch (Exception e)
    {
      StringBuffer b = new StringBuffer(200);
      b.append("Could not attach child object: ").append(child.toString());
      b.append(" to parent: ").append(parent.toString());
      b.append(". Root cause: ");
      b.append(e.getClass().getName()).append(": ").append(e.getMessage());
      log( context, Level.SEVERE, b.toString() );
    }
  }

  protected void attachData(
    Object parent,
    String data,
    ElementEndContext context)
  {
    // Find the appropriate setXXX(String) method.
    Method m = findMethod(context, parent, context.getElementName().getLocalName(), 
      sStringParamTypes);
    
    if (m == null)
      return;
        
    // Invoke the method.
    try
    {
      m.invoke(parent, new Object[] { data });
    }
    catch (InvocationTargetException ite)
    {
      log( context, Level.SEVERE,
        "Unable to attach data '" + data + "' for simple element " +
        context.getElementName().getLocalName() );
      ite.getTargetException().printStackTrace();
    }
    catch (Exception e)
    {
      log( context, Level.SEVERE, "Unable to attach data '" + data +
        "' for simple element " + context.getElementName().getLocalName());
      e.printStackTrace();
    }
  }

  protected void invokePreInitialize(Object parent, Object target)
  {
    // TODO: Define a preinit hook?
    // E.g: ((InitializableXMLObject)target).preInit(parent);
  }
  
  protected void invokePostInitialize(Object target)
  {
    // TODO: Define a postInit hook.
    // E.g: ((InitializableXMLObject)target).postInit();
  }
  
  protected Method findMethod(ElementContext context, 
    Object o, String elementName, Class[] paramTypes)
  {
    try
    {
      return findMethodEx(o, elementName, paramTypes);
    }
    catch (NoSuchMethodException nme)
    {
      StringBuilder sb = new StringBuilder("(");
      for (int i = 0; i < paramTypes.length; i++)
      {
        if (i > 0)
          sb.append(",");
        sb.append(paramTypes[i].getName());          
      } 
      sb.append(")");
      log( context, Level.SEVERE, "Class " + o.getClass().getName() +
        " has no set or add method for element named " + elementName + 
        " matching signature " + 
        getMethodName(SET_METHOD_PREFIX, elementName) + sb.toString() + " or " + 
        getMethodName(ADD_METHOD_PREFIX, elementName) + sb.toString() );
      nme.printStackTrace();
      return null;
    }    
  }
  
  private final Method findMethodEx(Object o, String elementName, Class[] paramTypes) 
      throws NoSuchMethodException
  {
    try
    {
      return o.getClass().getMethod(getMethodName(SET_METHOD_PREFIX, elementName), paramTypes);
    }
    catch (NoSuchMethodException nsme)
    {
      return o.getClass().getMethod(getMethodName(ADD_METHOD_PREFIX, elementName), paramTypes);
    }
  }
  
  protected String getMethodName(String prefix, String elementName)
  {
    char[] name = new char[prefix.length() + elementName.length()];
    for (int i = 0; i < prefix.length(); i++)
    {
      name[i] = prefix.charAt(i);
    }
    boolean nextUpper = true;
    int nameIndex = prefix.length();
    for (int i = 0; i < elementName.length(); i++)
    {
      if (Character.isJavaIdentifierPart(elementName.charAt(i)))
      {
        if (nextUpper)
        {
          name[nameIndex] =
            Character.toUpperCase(elementName.charAt(i));
          nextUpper = false;
        }
        else
        {
          name[nameIndex] = elementName.charAt(i);
        }
        nameIndex++;
      }
      else
      {
        nextUpper = true;
      }
    }
    return new String(name).trim();
  }

  /**
   * An object which can resolve an ElementName into a Class type for that
   * element.
   */
  public static interface ElementTypeResolver 
  {
    /**
     * Resolves a fully-qualified element name to a runtime type which is capable
     * of consuming information in child elements as laid out by the rules
     * defined by {@link DynamicHook}.
     * 
     * @param  elementName the fully qualified name of the element to be
     *    resolved.
     * @return a {@link java.lang.Class} representing the runtime type capable
     * of consuming the xml, or <code>null</code> if no matching type is found.
     */
    public Class resolveType( ElementName elementName );
  } 
  
  private static final class StackObject
  {
    private final Object _object;
    private final DynamicMetaClass _dynamicMetaClass;

    StackObject(Object object)
    {
      this(object, null);
    }
    
    StackObject(DynamicMetaClass dynamicMetaClass)
    {
      this(null, dynamicMetaClass);
    }
    
    private StackObject(Object object, DynamicMetaClass dynamicMetaClass)
    {
      _object = object;
      _dynamicMetaClass = dynamicMetaClass;      
    }
    
    final Object getObject()
    {
      return _object;
    }
    
    final DynamicMetaClass getDynamicMetaClass()
    {
      return _dynamicMetaClass;
    }
  }
  
  private final class DynamicMetaClass<T> extends MetaClass<T>
  {
    private final DeferredElementVisitorHook _deferredHook = new DeferredElementVisitorHook();
    private final Object _parent;
    
    DynamicMetaClass(MetaClass<T> metaClass, Object parent)
    {
      super(metaClass.getClassLoader(), metaClass.getClassName());
      _parent = parent;
    }
    
    final DeferredElementVisitorHook getDeferredHook()
    {
      return _deferredHook;
    }
    
    final Object getParent()
    {
      return _parent;
    }
    
    @Override
    public boolean equals(Object object)
    {
      if ( object == this )
      {
        return true;
      }
      if ( !(object instanceof DynamicMetaClass) )
      {
        return false;
      }   
      DynamicMetaClass dc = (DynamicMetaClass)object;
      
      return nullSafeEquals(dc.getClassLoader(), this.getClassLoader()) &&
             nullSafeEquals(dc.getClassName(), this.getClassName()) &&
             nullSafeEquals(dc._parent, this._parent);        
    }
    
    private final boolean nullSafeEquals(Object o1, Object o2)
    {
      if (o1 == null && o2 == null)
        return true;
      if (o1 == null || o2 == null)
        return false;
      return o1.equals(o2);
    }

    @Override
    public T newInstance() 
      throws InstantiationException, IllegalAccessException, ClassNotFoundException
    {
      _handlerLock.lock();
      try
      {
        _applicationObjectStack.push(new StackObject(getParent()));
        
        ExtensionRegistry extensionRegistry = ExtensionRegistry.getExtensionRegistry();
        DefaultElementContext context = (DefaultElementContext) extensionRegistry.createInitialContext();

        _deferredContext = new DeferredContext(this, context);
        
        _deferredHook.attachElementVisitor(DynamicHook.this);

        _deferredContext.rethrowExceptions();
        
        return (T)_deferredContext.getApplicationObject();
      }
      finally
      {
        _applicationObjectStack.pop();
        
        _deferredContext = null;
        _handlerLock.unlock();
      }
    }
    
    @Override
    public Class toClass() 
    {
      // Deny access to unsupported constructors.
      throw new UnsupportedOperationException("Access to Class of dynamic MetaClass illegal");
    }
  }  
  
  private static final class DeferredContext
  {
    private final DynamicMetaClass _dynamicMetaClass;
    private final ElementContext _context;
    private Object _applicationObject;
    private ExceptionWrapper _exceptionWrapper;

    DeferredContext(DynamicMetaClass dynamicMetaClass, ElementContext context)
    {
      _dynamicMetaClass = dynamicMetaClass;
      _context = context;
    }
    
    final DynamicMetaClass getDynamicMetaClass()
    {
      return _dynamicMetaClass;
    }
    
    final void setApplicationObject(Object applicationObject)
    {
      _applicationObject = applicationObject;
    }
    
    final Object getApplicationObject()
    {
      return _applicationObject;
    }
    
    final void setInstantiationException(InstantiationException ie)
    {
      setFinalExceptionWrapper(new ExceptionWrapper(ie));
    }
    
    final void setIllegalAccessException(IllegalAccessException iae)
    {
      setFinalExceptionWrapper(new ExceptionWrapper(iae));
    }
    
    final void setClassNotFoundException(ClassNotFoundException cnfe)
    {
      setFinalExceptionWrapper(new ExceptionWrapper(cnfe));
    }
    
    final void setFinalExceptionWrapper(ExceptionWrapper exceptionWrapper)
    {
      if (_exceptionWrapper == null)
        _exceptionWrapper = exceptionWrapper;
    }
    
    final void rethrowExceptions()
      throws InstantiationException, IllegalAccessException, ClassNotFoundException
    {
      if (_exceptionWrapper == null)
        return;      
      if (_exceptionWrapper._ie != null)
        throw _exceptionWrapper._ie;
      if (_exceptionWrapper._iae != null)
        throw _exceptionWrapper._iae;
      if (_exceptionWrapper._cnfe != null)
        throw _exceptionWrapper._cnfe;
    }
    
    private class ExceptionWrapper
    {
      private InstantiationException _ie;
      private IllegalAccessException _iae;
      private ClassNotFoundException _cnfe;
      
      ExceptionWrapper(InstantiationException ie)
      {
        _ie = ie;
      }

      ExceptionWrapper(IllegalAccessException iae)
      {
        _iae = iae;
      }

      ExceptionWrapper(ClassNotFoundException cnfe)
      {
        _cnfe = cnfe;
      }
    }
  }
}
