/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.editor;

import java.util.EventListener;

/**
 * The <code>EditorListener</code> interface should be
 * implemented by clients that are interested in receiving
 * notifications about <code>Editor</code> instances being opened,
 * activated, deactivated, or closed in the IDE. <p>
 *
 * This allows clients to attach to or detach from the given Editor
 * for the purpose of providing extra functionality.  For example, the
 * debugger may use this for determining when a Code Editor is opened
 * so that it can display the breakpoint icon locations properly. <p>
 *
 * Clients should register listeners with the EditorManager singleton
 * instance.  When an editor is opened, the <code>opened()</code>
 * method will be called.  When an editor is about to be closed, the
 * <code>closed()</code> method will be called. <p>
 *
 * @see Editor
 * @see EditorManager
 * @see EditorEvent
 */
public interface EditorListener extends EventListener
{
  /**
   * Notify listeners that a new Editor has been opened.  This method
   * is called after an Editor is opened, and its <code>open()</code>
   * method called.
   * @param event the {@link EditorEvent} object.
   */
  public void opened( EditorEvent event );

  /**
   * Notify listeners that an existing Editor has been activated.
   * This method is called after an Editor is activated, and its
   * <code>activate()</code> method called.
   * @param event the {@link EditorEvent} object.
   */
  public void activated( EditorEvent event );

  /**
   * Notify listeners that an existing Editor listener is
   * de-activated.  This methodis called after an Editor is
   * deactivated, and its <code>deactivate()</code> method called.
   * @param event the {@link EditorEvent} object.
   */
  public void deactivated( EditorEvent event );

  /**
   * Notify listeners that an existing Editor is being closed.  This
   * method is called right before an Editor's <code>close()</code>
   * method is called.
   * @param event the {@link EditorEvent} object.
   */
  public void closed( EditorEvent event );
}
