/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.debug;

import java.io.IOException;

/**
 * ServerConnectors allow clients to start the debugger listening by calling
 * the method {@link ServerConnector#startListening()} waiting for a debuggee
 * process to start. The debugge process can be started by the extension or
 * the IDE. When the debuggee process is launched, the listening debugger will 
 * automatically accept the connection.
 * 
 * IDE providers must provide an implementation of this interface. Extension
 * writers can access the <code>Connector</code> through the 
 * {@link Debugger#getServerConnector(Context)} method. 
 */
public abstract class ServerConnector extends Connector
{
  private boolean _isListening = false;

  /**
   * Tell the debugger to start listening for and accept a connection
   * from a debuggee. This method will return after the debugger
   * starts listening, but before it actually accepts a connection.
   * (The debugger will create and start a new thread to accept
   * the connection). After the debugger accepts a connection, it
   * will automatically stop listening for subsequent connections.
   *
   * After calling this method, the extension writer should either start the
   * debuggee process or tell the IDE to do so by calling {@link
   * Connector#startDebuggee(Map)}.
   * 
   * To tell the debugger to stop listening before it accepts a
   * connection, (for example, if the extension was unable to start the
   * debuggee process successfully), the extension should call
   * {@link #stopListening()}.
   *
   * If this ServerConnector has already accepted a connection (and
   * has automatically stopped listening), it may or may not be able
   * to start listening again. If it does not support listening for
   * another connection, this method will throw
   * <code>UnsupportedOperationException</code>.
   * 
   * @exception IOException if this connector is unable to start
   * listening.
   * @exception UnsupportedOperationException if this connector has
   * already accepted a connection and does not support listening for
   * another connection.
   */
  public final void startListening() throws IOException
  {
    try
    {
      startListeningImpl();
    }
    finally
    {
      _isListening = true;
    }
  }
    
  /**
   * IDE service providers implement this method to start listening.
   * 
   * @throws IOException if this connector is unable to start listening.
   */
  protected abstract void startListeningImpl() throws IOException;
  
  /**
   * Check if the debugger is listening for a debuggee connection.
   * @return true if listening; false otherwise.
   */
  public final boolean isListening()
  {
    return _isListening;
  }
  
  /**
   * Tell the listening debugger to stop listening.
   */
  public final void stopListening()
  {
    try
    {
      stopListeningImpl();
    }
    finally
    {
      _isListening = false;
    }
  }
  
  /**
   * IDE service providers implement this method to stop listening.
   */
  protected abstract void stopListeningImpl();
}

