/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.debug;

import java.io.IOException;
import java.util.Map;

import javax.ide.command.Context;
/**
 * The <code>Connector</code> interface is used to connect to a debugging 
 * session. The <code>Connector</code> object is used to start the debuggee 
 * or to retrieve command line options so the extension can start the debuggee 
 * itself.<p>
 * 
 * IDE providers must provide an implementation of this interface. Extension
 * writers can access the <code>Connector</code> through the 
 * {@link Debugger#getClientConnector(Context)} and 
 * {@link Debugger#getServerConnector(Context)} 
 * methods. 
 */
public abstract class Connector 
{
  /**
   * The key of the option in the options map for the absolute
   * path of the Java executable to run.
   */
  public static final String OPTION_JAVA_EXECUTABLE = "java-executable";
  
  /**
   * The key of the option in the options map for an argument
   * passed to Java to set the VM. This is normally the first option passed
   * to Java, e.g. for the client VM, this would be String[0] or 
   * { "-client' }.
   */
  public static final String OPTION_JVM = "jvm";
  
  /**
   * The key of an option in the options map for zero or more
   * arguments which must be passed into java immediately following the 
   * switch specifying the VM. These options are specific to the debugger.
   */
  public static final String OPTION_DEBUG_FIRST = "debug-first";
  
  /**
   * The key of an option in the options map for arguments
   * required to set the classpath of Java. For example, a typical value
   * might be { "-classpath", "somejar.jar" }
   */
  public static final String OPTION_CLASSPATH = "classpath";
  
  /**
   * The key of an option in the options map for options that
   * should be passed into the java exectuable.
   */
  public static final String OPTION_JAVA_OPTIONS = "java-options";
  
  /**
   * The key of an option in the options map for options that are passed 
   * into Java immediately before the main class. These options are specific
   * to the debugger.
   */
  public static final String OPTION_DEBUG_LAST = "debug-last";
  
  /**
   * The key of an option in the options map for the main class to run.
   */
  public static final String OPTION_MAIN_CLASS = "main-class";
  
  /**
   * The key of an option in the options map for arguments passed into the
   * java process.
   */
  public static final String OPTION_PROGRAM_ARGUMENTS = "program-arguments";
  
  /**
   * Get the options the extension should use when starting the
   * debuggee.
   * 
   * The extension will mostly call this method when it wants to
   * start the debuggee process itself.
   * 
   * If the extension wants the IDE to start the debuggee process,
   * this method does not need to be called, unless the extension
   * wants to modify some of the command line options.
   * 
   * Returns a Map where the keys are the following String
   * constants
   * {@link #OPTION_JAVA_EXECUTABLE},
   * {@link #OPTION_JVM},
   * {@link #OPTION_DEBUG_FIRST},
   * {@link #OPTION_CLASSPATH},
   * {@link #OPTION_JAVA_OPTIONS},
   * {@link #OPTION_DEBUG_LAST},
   * {@link #OPTION_MAIN_CLASS},
   * {@link #OPTION_PROGRAM_ARGUMENTS},
   * and the values are String[].
   * Each String[] value can be null or of zero length, if no options are
   * necessary, or an array of one or more Strings.
   * 
   * @return a Map of command line options.
   */
  public abstract Map getOptions();
  
  /**
   * Start the debuggee process.
   * The extension should call this method when it wants the IDE to
   * start the debuggee process.
   * 
   * The <code>options</code> can be a full set of command line
   * options as returned {@link #getOptions()}, or it can be a
   * partial set of options. If the <code>options</code> map
   * contains a key/value pair, the value specified will override
   * the value that the IDE would ordinarily use for that key.
   * If the extension does not want to override any options, the
   * <code>options</code> parameter can be null.
   *
   * For example, if a extension wants to add an additional program
   * argument, it should call {@link #getOptions()} and get the
   * value for {@link #OPTION_PROGRAM_ARGUMENTS} out of the Map.
   * It should then create a new String[] to hold the original
   * program arguments plus the additional program argument. It
   * should then put this new String[] into the Map before
   * passing it to <code>startDebuggee</code>.
   *
   * @param options a Map of command line options
   * @return the debuggee process
   * @exception IOException if the process can not be started.
   */
  public abstract Process startDebuggee( Map options ) throws IOException;
  
  /**
   * Check if the debugger has connected to the debuggee.
   * @return true if connected.
   */
  public abstract boolean isConnected();
  
  /**
   * Check if the debugger has finished debugging.
   * @return true if finished.
   */
  public abstract boolean isFinished();
  
  /**
   * Get the {@link Context} used to create this connector.
   * @return the context.
   */
  public abstract Context getContext();
}

