/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.build;

import java.util.EventListener;

/**
 * Extension writers request notification of {@link BuildSystem} events by 
 * implementing this interface and registering their build listener with the 
 * extension deployment descriptor.<p>
 *
 * IDE providers are responsible for calling the <code>BuildListener</code>
 * methods before their compiler starts compiling documents and immediately
 * after it finishes doing so.
 */
public interface BuildListener extends EventListener
{
  /**
   * The <code>preBuild</code> method is called before documents are compiled. 
   * If a listener throws an {@link AbortBuildException}, the entire build is 
   * aborted and the <code>postBuild</code> methods are called immediately on 
   * the listeners whose <code>preBuild</code> method had a chance to 
   * execute (incuding the one that aborted the build).  <p>
   * The <code>postBuild</code> method is not called on listener 
   * that did not execute their <code>preBuild</code> method. <p>
   * <p>
   * This method may be called from threads other than the AWT event thread.
   * 
   * @param event the build event.
   * @throws javax.ide.build.AbortBuildException to abort the entire build.
   */
  public void preBuild( BuildEvent event ) throws AbortBuildException;

  /**
   * When the compiler finishes compiling documents, the 
   * <code>postBuild</code> methods are called. If any <code>postBuild</code>
   * method throws an {@link AbortBuildException}, the <code>postBuild</code> 
   * methods are still called on the remaining listeners. Implementors of the 
   * <code>postBuild</code> method must always check the 
   * {@link BuildEvent#isBuildSuccessful} method before continuing. <p>
   * <p>
   * This method may be called from threads other than the AWT event thread.
   * 
   * @param event the build event.
   * @throws javax.ide.build.AbortBuildException to abort the entire build.
   */
  public void postBuild( BuildEvent event ) throws AbortBuildException;
}

